/*
* \file: errmemd_backend_block_device.c
*
* Definition of a backend which represents a block device as backend.
* 
* Definition of the functions provided by the block device backend.
*
* \component: errmemd
*
* \author: Kai Tomerius (ktomerius@de.adit-jv.com)
*          Markus Kretschmann (mkretschmann@de.adit-jv.com)
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* Implementation of an error memory backend that stores error memory
* messages on a block device persistent storage. Access is
* byte-oriented, i.e. only modified or new data  is  written to persistent
* storage.
*
* The first blocks can be a number of non-overwritable blocks, followed
* by a number of overwritable blocks. When the last block is reached,
* writing wraps around to the first overwritable block. The number of
* non-overwritable blocks can be configured with the persistent parameter.
* If not provided or zero all available blocks will be overwritten.
*
*                   | 0...<p> | <p>+1...<n> |
*
*                   where p >= 0
*
* The first block keeps the information described in the struct
* ErrmemBackendBlockDeviceHeader_t about the persistent memory
* at its beginning, followed by the block header and the data
*
* First block:
*
* | BlockDeviceHeader | BlockDeviceBlock | 1 ... <n> |
* 
*   where n = 4Byte message length + message
*
* All other blocks 
*
* | BlockDeviceBlock | 1 ... <n> |
* 
*   where n = 4Byte message length + message
*
* Each blocks has a unique, monotonically increasing number
* <blocknum> and can hold one or more error memory messages.
*
*
* \history
* <history item>
*/

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include <ctype.h>
#include <regex.h>
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <linux/fs.h>
#include <linux/limits.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/sendfile.h>
#include <syslog.h>
#include <mtd/mtd-user.h>
#include <errno.h>

#include "errmemd.h"
#include "errmem_backend.h"
#include "errmem_socket_interface.h"

#define STD_SECTOR_SIZE   512

/* Information of the persistent memory resides at the beginning of
 * the first block.
 * Take care that this header (ErrmemBackendBlockDeviceHeader_t)
 * is always 4 byte aligned.
 */
#define MAGIC_HEADER 0x22d9f147 // random number
typedef struct  __attribute__ ((packed)) ErrmemBackendBlockDeviceHeader {
	uint32_t magic;           //header magic
	uint16_t crc;             //header crc
	uint16_t header_length;   //total length of the current header
	uint32_t blocksize;       //blocksize with which the storage is created
	uint32_t persistent;      //number of persistent (non-overwritable) blocks
	uint32_t blocks;          //total number of blocks available in the storage
	uint32_t blockmagic;      //keeps the valid block magic for the current persistent storage
	uint32_t w_width;         //alignment with which this storage was created
} ErrmemBackendBlockDeviceHeader_t;

/* subsequent block of the persistent memory
 * Take care that this header is 4 byte aligned including the data
 * field blocknum 
 */
#define MAGIC_BLOCK_ZERO 0xbeefbeaf //random number
#define MAGIC_BLOCK_INVALID 0xFFFFFFFF
typedef struct  __attribute__ ((packed)) ErrmemBackendBlockDeviceBlock {
	uint32_t magic;
	uint16_t crc;
	uint16_t reserved;
	uint32_t blocknum;
	unsigned char data [1];  // error memory messages
} ErrmemBackendBlockDeviceBlock_t;

// the error memory backend
typedef struct ErrmemBackendBlockDevice {
	ErrmemBackend_t backend;     // error memory backend interface
	int32_t  type;               // type of backend storage
	int32_t  access_mode;        // read/write access
    int32_t  access_options;     // create or backup or nothing of these
	uint32_t status;             // status of the storage
	uint32_t erasesize;          // erasesize in case of flash device
	uint32_t blocksize;          // size of one block in bytes
	uint32_t blocks;             // number of blocks
	int32_t  stat_mode;          // mode flags of device from system view
	uint32_t stat_blocksize;     // size of one block from system view
	uint32_t stat_blocks;        // number of blocks from system view
	uint32_t stat_size;          // size of device from system view
	uint32_t persistent;         // number of blocks which are not overwritten
	uint32_t blocknum;           // monothonically block number of the currently used block
	uint32_t seqnum;             // message internal sequence number
	uint32_t flags;              // flags of the last dumped message
	uint32_t storage;            // monothonically increasing index of message
	int32_t  fd;                 // file handle of the block device
	uint32_t writepos;           // write position inside the current block
	uint32_t writepos_absolute;  // absolute write position at startup (needed in client mode only)
	uint32_t last_stored;        // sequence number of last stored message in persistent storage
	uint32_t blockmagic;         // block magic for the current block device
	uint32_t msg_magic;          // current message magic valid for current storage
	uint32_t w_width;            // alignment for write requests to comply with 
	uint16_t seed;               // seed value for crc-calculation
	uint32_t used;               // used blocks when device is set up (only for information purpose)
	uint32_t nr;                 // position of device in command line
	int32_t  def;                // marks this backend to be the default one or a backup
	uint32_t wait;               // wait time between two erase cycles of the same physical eraseblock
	uint32_t retry;              // number of retries if error occurs on preparing a NOR flash block
	char    *name;               // file name of the block device
	char    *command;            // command to execute when creating the underlying persistent storage
	ErrmemBackendBlockDeviceHeader_t* header; //meta information
	ErrmemBackendBlockDeviceBlock_t* block;  // current block
	uint64_t *times;             // last erase time of a bare NOR flash erase block
} ErrmemBackendBlockDevice_t;

static uint32_t getCurrentMsgMagic(uint32_t in)
{
	uint64_t base = ((uint64_t)((uint64_t)in & 0x00000000FFFFFFFF)) << 32;
	base |= (uint64_t)((uint64_t)MAGIC_MESSAGE & 0x00000000FFFFFFFF);
	base = (~base) + (base << 18);
	base = base ^ (base >> 31);
	base = base * 21;
	base = base ^ (base >> 11);
	base = base + (base << 6);
	base = base ^ (base >> 22);
	return (uint32_t) base;
}

static inline struct PersistentMessage* mkpmsg(unsigned char* m, uint32_t o)
{
	return (struct PersistentMessage*)addptr(m, o);
} 

static inline uint32_t* mkuint32(unsigned char* m, uint32_t o)
{
	return (uint32_t*)addptr(m, o);
} 

static inline struct PersistentMessage_old* mkpmsgo(unsigned char* m, uint32_t o)
{
	return (struct PersistentMessage_old*)addptr(m, o);
} 

// mkmsg - cast to struct errmem_message* (and avoid a lint warning)
static inline struct errmem_message* mkmsg(unsigned char* m, uint32_t o)
{
	return (struct errmem_message*) addptr(m, o);
}

static inline uint32_t aligned_len(uint32_t len, uint32_t align)
{
	uint32_t al = 0;
	if (!align || !~align)
		al = len;
	else if ( (len % align) == 0)
		al = len;
	else
		al = len + align - len % align;
	return al;
}

static int32_t read_dev_header(ErrmemBackendBlockDevice_t* dev)
{
	int32_t err = 0;
	uint32_t al = 0;
	if (dev && dev->name){
		if(!dev->header){
			al = aligned_len(sizeof(ErrmemBackendBlockDeviceHeader_t), dev->w_width);
			dev->header = (ErrmemBackendBlockDeviceHeader_t*)get_mem(al);
			if (dev->header)
				memset(dev->header, 0xFF, al);
		}
		if (dev->header){
			err = seek_dev(dev->fd, dev->name, 0, SEEK_SET);	
			if(0 == err) {
				err = read_dev(dev->fd, dev->name, (char*)dev->header,
							   sizeof(ErrmemBackendBlockDeviceHeader_t));
				if (err < 0) {
					syslog(LOG_CRIT, "%s %s %s %s",
						   "read_dev_header: read failed on device ",
						   dev->name, "error = ", strerror(-err));
				} else {
					err = seek_dev(dev->fd, dev->name, 0, SEEK_SET);	
					if (0 != err)
						err = -errno;
				}
			} else
				syslog(LOG_CRIT, "%s %s %s %s",
					   "read_dev_header: failed to seek to beginning of storage",
					   dev->name, " error = ", strerror(-err));
		} else
			err = -ENOMEM;
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s", "read_dev_header: parameter check for dev failed");
	}
	return err;
}

static int32_t check_set_device_type(ErrmemBackendBlockDevice_t* dev)
{
	int32_t err = 0;
	
	if (dev) {
		err = (dev->stat_mode & S_IFMT);
		if ((err == UNDEFINED) || (err == S_IFIFO) || 
			(err == S_IFDIR) || (err == S_IFSOCK)) {
			syslog(LOG_CRIT, "%s %s %s %s %s %s",
				   "check_set_device_type (block device backend): "
				   "stat delivered not supported type: ",
				   int_stat_type2char_stat_type(err),
				   " for device: ", dev->name, " error = ",
				   strerror(EMEDIUMTYPE));
			err = -EMEDIUMTYPE;
		}

		if ((err > 0) && (err != S_IFLNK)) {
			if ((dev->access_mode & O_RDWR)) {
				if (((dev->type == BLOCK) || (dev->type == BLOCK_FLASH) || 
					 (dev->type == BLOCK_NOR) || (dev->type == BLOCK_NAND)) && 
					(err != S_IFREG))
					err = -EINVAL;
				else if ((dev->type == BLOCK_RAW) && (err != S_IFBLK))
					err = -EINVAL;
				else if (((dev->type == BLOCK_FLASH_RAW) || (dev->type == BLOCK_NOR_RAW)) &&
						 (err != S_IFBLK) && (err != S_IFCHR))
					err = -EINVAL;
				else if (dev->type == BLOCK_NAND_RAW)
					err = -EMEDIUMTYPE;
				else if (((dev->type == BLOCK_RAW) || (dev->type == BLOCK_FLASH) ||
						  (dev->type == BLOCK_FLASH_RAW) || (dev->type == BLOCK_NOR) || 
						  (dev->type == BLOCK_NOR_RAW) || (dev->type == BLOCK_NAND)) &&
						 ((dev->access_options) & (OPT_ACC_CREATE | OPT_ACC_BACKUP)))
					err = -EINVAL;
			
				if (err < 0)
					syslog(LOG_CRIT, "%s %s %s %s",
						   "check_set_device_type (block device backend): "
						   "conflicting device type, stat_mode, access options "
						   "for device: ", dev->name,
						   " error = ", strerror(-err));
			} else {
				if (err == S_IFREG) {
					if (dev->type == BLOCK_RAW)
						dev->type = BLOCK;
					else if (dev->type == BLOCK_FLASH_RAW)
						dev->type = BLOCK_FLASH;
					else if (dev->type == BLOCK_NOR_RAW)
						dev->type = BLOCK_NOR;
					else if (dev->type == BLOCK_NAND_RAW)
						dev->type = BLOCK_NAND;
				}
				if (dev->access_options) {
					dev->access_options = OPT_ACC_NONE;
					syslog(LOG_CRIT, "%s %s", "check_set_device_type (block device backend): "
						   "access options in client mode not allowed - set to nothing for device: ", dev->name);
				}
			}
		}
	} else {
			syslog(LOG_CRIT, "%s", "check_set_device_type (block device backend): "
			   " Parameter check for dev failed");
		err = -EINVAL; 
	}
	return err;
}

static int32_t stat_dev(ErrmemBackendBlockDevice_t* dev)
{
	int32_t err  = 0;
	int32_t llen = 0;
	char* name   = NULL;
	char* c_name = NULL;
	struct stat buf = {0};

	if (dev && dev->name) {
		name = strndup(dev->name, strlen(dev->name) + 1);
		if (!name)
			err = -errno;
		else
			err = S_IFLNK;
		while (err == S_IFLNK) {
			err = stat(name, &buf);
			if (-1 == err)
				err = -errno;
			else {
				dev->stat_mode = buf.st_mode;
				err = check_set_device_type(dev);
				if (err > 0) {
					if (err != S_IFLNK) {
						dev->stat_blocksize = buf.st_blksize;
						dev->stat_blocks    = buf.st_blocks;
						dev->stat_size      = buf.st_size;
					} else {
						c_name = (char*)get_mem(buf.st_size + 1);
						if (!c_name)
							err = -ENOMEM;
						else {
							llen = readlink(name, c_name, buf.st_size + 1);
							if (!~llen)
								err = -errno;
							if (llen > buf.st_size)
								err = -ENAMETOOLONG;
							free(name);
							name = NULL;
							if (!err) {
								*(c_name + buf.st_size) = '\0';
								name = strndup(c_name, strlen(c_name) + 1);
								if (!name)
									err = -errno;
								free(c_name);
								c_name = NULL;
							}
						}
					}
				}
			}
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s", "stat_dev: parameter check for dev failed");
	}
	if (c_name)
		free(c_name);
	if (name)
		free(name);
	return err;
}

static int32_t get_device_dimensions(ErrmemBackendBlockDevice_t* dev)
{
	int32_t err = 0;
	uint32_t  n = 0;
	mtd_info_t mtd_info = {0};
	if (dev && dev->name) {
		if (dev->type == BLOCK_RAW) {
			err = ioctl(dev->fd, BLKBSZGET, &n);
			if ((-1 != err) && (n > 0)) {
				dev->stat_blocksize = n;
				err = ioctl(dev->fd, BLKGETSIZE, &n);
				if ((-1 != err) && (n > 0))
					dev->stat_blocks = n;
				else {
					err = -errno;
					syslog(LOG_CRIT, "%s %s %m", "get_device_dimensions (flash device): "
				   "ioctl call BLKGETSIZE failed for device ", dev->name);
				}
			} else {
				err = -errno;
				syslog(LOG_CRIT, "%s %s %m", "get_device_dimensions (flash device): "
					   "ioctl call BLKBSZGET failed for device ", dev->name);
			}
		} else if ((dev->type == BLOCK_FLASH) ||
				   (dev->type == BLOCK_FLASH_RAW)) {
			err = ioctl(dev->fd, MEMGETINFO, &mtd_info);
			if (-1 != err) {
				if (mtd_info.type == MTD_NORFLASH) {
					if (dev->type == BLOCK_FLASH_RAW)
						dev->type = BLOCK_NOR_RAW;
					else if (dev->type == BLOCK_FLASH)
						dev->type = BLOCK_NOR;
				} else if (mtd_info.type == MTD_NANDFLASH) {
					if (dev->type == BLOCK_FLASH_RAW)
						dev->type = BLOCK_NAND_RAW;
					else if (dev->type == BLOCK_FLASH)
						dev->type = BLOCK_NAND;
				} else {
					err = -EMEDIUMTYPE;
						syslog(LOG_CRIT, "%s %s %s %m", "get_device_dimensions (flash device): ",
							   dev->name, strerror(EMEDIUMTYPE));
				}
				if (dev->type == BLOCK_NAND_RAW) {
					err = -ENOSYS;
					syslog(LOG_CRIT, "%s %s %s %m", "get_device_dimensions (flash device) - "
						   " Raw NAND flash access is not accepted by the daemon ",
						   dev->name, strerror(ENOSYS));
				}
				if (!err) {
					/* Set erasesize in read only mode too - we can issue command erase */
					dev->stat_size = mtd_info.size;
					dev->stat_blocksize = 0;
					dev->stat_blocks    = 0;
					dev->erasesize = mtd_info.erasesize;

					/* check for ratio between bloscksize and erasesize has to done only if write access */
					if (dev->access_mode & O_RDWR) {
						if (dev->blocksize < dev->erasesize) {
							err = -EINVAL;
							syslog(LOG_CRIT, "%s %s %s %s %s %d %s %d %m", "get_device_dimensions (flash device): ",
								   dev->name, " blocksize specified is less than erasesize of flash device ",
								   strerror(EINVAL), "blocksize : erasesize", dev->blocksize, " : ", dev->erasesize);
						} else if ((dev->blocksize % dev->erasesize)) {
							err = -EINVAL;
							syslog(LOG_CRIT, "%s %s %s %s %s %d %s %d %m", "get_device_dimensions (flash device): ",
								   dev->name, " blocksize specified is not a multiple of the erasesize of flash device ",
								   strerror(EINVAL), "blocksize : erasesize", dev->blocksize, " : ", dev->erasesize);
						}
					}
				}
			} else {
				err = -errno;
				syslog(LOG_CRIT, "%s %s %m", "get_device_dimensions (flash device): "
				   "ioctl call MEMGETINFO failed for device ", dev->name);
			}
		}
	    if (err) {
			dev->stat_blocksize =  0;
			dev->stat_blocks    =  0;
			dev->stat_size      =  0;
			dev->erasesize      = ~0;
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s", "get_device_dimensions (block device): parameter check for dev failed");
	}
	return err;
}


// get_blocknum - return blocknum of the n-th block if valid or MAGIC_BLOCK_INVALID
static uint32_t get_blocknum(
	ErrmemBackendBlockDevice_t* dev, uint32_t n)
{
	uint32_t blocknum = 0;
	int32_t  err = 0;
	off_t    off = 0;

	if (dev && dev->block && dev->header) {
		if (!n) {
			/* Check block zero. Take device header into account */
			off = (off_t)aligned_len(dev->header->header_length, dev->w_width);
		}

		off += (off_t)(n*dev->blocksize);

		err = seek_dev(dev->fd, dev->name, off, SEEK_SET);
		if (err < 0) {
			blocknum = MAGIC_BLOCK_INVALID;
		}
		else {
			// read the block and check the crc
			uint16_t crc;
			err = read_dev(dev->fd, dev->name, (char*)dev->block, sizeof(ErrmemBackendBlockDeviceBlock_t));
			if (err < 0)
				blocknum = MAGIC_BLOCK_INVALID;

			if (!blocknum) {
				//check first the block magic and only if they are identical then check whether the
				//block is consistent. An invalid block magic may fit to an crc which was calculated in
				//former runs with this former valid block magic.
				if (((ErrmemBackendBlockDeviceBlock_t*)(dev->block))->magic != dev->header->blockmagic)
					blocknum = MAGIC_BLOCK_INVALID;
				if (!blocknum) {
					//at last check the block crc
					crc = ((ErrmemBackendBlockDeviceBlock_t*)(dev->block))->crc;
					((ErrmemBackendBlockDeviceBlock_t*)(dev->block))->crc = 0;

					if (calc_crc16(dev->seed,
								   (void*)((ErrmemBackendBlockDeviceBlock_t*)(dev->block)),
								   sizeof(ErrmemBackendBlockDeviceBlock_t))==crc) {
						((ErrmemBackendBlockDeviceBlock_t*)(dev->block))->crc = crc;
						blocknum = ((ErrmemBackendBlockDeviceBlock_t*)(dev->block))->blocknum;
					} else 
						blocknum = MAGIC_BLOCK_INVALID;
				}
			}
		}
	} else {
		blocknum = MAGIC_BLOCK_INVALID;
		syslog(LOG_CRIT, "%s %s", "get_blocknum: Parameter check failed: error = ", strerror(EINVAL));
	}
		
	return blocknum;
}

static uint32_t get_write_position(struct ErrmemBackend* backend)
{
	uint32_t pos = 0;
	uint32_t begin_block = 1;
	PersistentMessage_t pmsg      = {0};
	PersistentMessage_old_t opmsg = {0};
	struct errmem_message* e_msg  = NULL;
	uint32_t wpos                 = 0;
	uint32_t clen                 = 0;
	uint32_t crc                  = 0;
	int32_t  found                = 0;
	uint16_t p_crc                = 0;
	uint32_t magic                = 0;
	uint32_t offset               = 0;
	uint32_t blocknum             = 0;
	uint32_t len                  = 0;
	uint32_t acc                  = 0;
	uint32_t hlen = sizeof(ErrmemBackendBlockDeviceBlock_t) - 1;
	ErrmemBackendBlockDevice_t* l = (ErrmemBackendBlockDevice_t*) backend;
	ErrmemBackendBlockDeviceBlock_t* block;
	if (l->block->magic == MAGIC_HEADER) {
		uint32_t llen = aligned_len(l->header->header_length, l->w_width);
		block = (ErrmemBackendBlockDeviceBlock_t*)addptr((unsigned char*)l->block, llen);
		hlen += llen;
	}
	else
		block = l->block;

	do {
		acc   = 0;
		found = 0;
		magic = *(mkuint32(&block->data[pos], 0));
		/* Check whether CRC for message header is supported or not */
		if (magic == MAGIC_MESSAGE) {
			/* This message format does not support CRC check but the
			 * message is a valid one 
			 */
			opmsg = *(mkpmsgo(&block->data[pos], 0));
			offset   = offsetof(PersistentMessage_old_t, msg);
			e_msg    = mkmsg(block->data, pos + offset);
			blocknum = opmsg.blocknum;
			len      = opmsg.len;
			acc = 1;
		} else if (magic == l->msg_magic) {
			pmsg = *(mkpmsg(&block->data[pos], 0));
			p_crc = pmsg.crc;
			pmsg.crc = 0;
			pmsg.crc = calc_crc16(l->seed, (void*)&pmsg, offsetof(PersistentMessage_t, msg));
			if (pmsg.crc == p_crc) {
				/* valid message with new format */
				offset   = offsetof(PersistentMessage_t, msg);
				e_msg    = mkmsg(block->data, pos + offset);
				blocknum = pmsg.blocknum;
				len      = pmsg.len;
				acc = 1;
			}
		}

		/* I we don't find a valid message at pos, we search for a possible next
		 * valid message
		 */
		if (acc) {
			/* It seems we have found a valid message which was
			 * written to this block with the current block number
			 */
			if (blocknum == block->blocknum)
				/* This message belongs to this current block */
				found = 1;
			else
				/* This message seems to be valid but does not belong to this
				 * block. We start to write next message from pos
				 */
				break;
		}
			
		if (found && ~len && len && e_msg) {
			/* check length against message length */
			if (len > sizeof(struct errmem_message))
				clen = sizeof(struct errmem_message);
			else
				clen = len;
			/* In case we have the old message magic we check the CRC.
			 * If the CRC check fails, we stop here and write the next
			 * message from pos. A fail of the CRC check indicates that
			 * the seed for CRC calculation has changed. This is a hint
			 * that the blockmagic has been changed and this current
			 * message was not written to this block during current
			 * life cycle of the persistent storage. 
			 * In case of the new message magic we don't need to check
			 * the message CRC here because we already did the header
			 * CRC check which had to have passed when we are here.
			 * We just calculate new pos and that's it.
			 */
			if (magic == MAGIC_MESSAGE) {
				crc = e_msg->internal.crc;
				e_msg->internal.crc = 0;
				e_msg->internal.crc = calc_crc16(l->seed, (void*)e_msg, clen);
				if (e_msg->internal.crc != crc)
					/* stop here because an old style message seems to be 
					 * not part of this life cycle of the persistent storage 
					 */
					break;
			}
				
			if (!begin_block)
				pos += aligned_len((offset + clen), l->w_width);
			else {
				pos = aligned_len((hlen + offset + clen), l->w_width) - hlen;
				begin_block = 0;
			}

			l->last_stored = e_msg->internal.seqnum;
			wpos = pos;
		} else {
			/* set position to a possible next index which has to be an aligned one
			 * hsize at the beginning of a block contains already at last byte si->pos = 0 
			 */
			if(!pos && l->w_width){
				pos = aligned_len(hlen, l->w_width) - hlen;
			} else
				pos += l->w_width;
		}

	} while (l->blocksize > pos + hlen);

	return wpos;
}

static int32_t generate_block_magic(struct ErrmemBackend* backend, int32_t num_iterations)
{
	int32_t err = 0;
	ErrmemBackendBlockDevice_t* dev = (ErrmemBackendBlockDevice_t*) backend;
	uint32_t old_blockmagic = dev->header->blockmagic;
	int32_t hd = open_dev("/dev/urandom", O_RDONLY, 0);
	if (hd >= 0) {
		do {
			num_iterations--;
			err = read_dev(hd, "/dev/urandom", (void*)&dev->blockmagic,
						   sizeof(dev->blockmagic));
		} while (((err >= 0) && ((old_blockmagic == dev->blockmagic)     ||
								 (MAGIC_HEADER == dev->blockmagic)       ||
								 (MAGIC_BLOCK_INVALID == dev->blockmagic)||
								 (MAGIC_BLOCK_ZERO == dev->blockmagic)   ||
								 (num_iterations > 0))));
		close(hd);
	} else
		err = hd;
	if (err >= 0)
		err = 0;

	return err;
}

static int32_t generate_new_magics(struct ErrmemBackend* backend) 
{
	int32_t err = 0;
	int32_t it  = 0;
	ErrmemBackendBlockDevice_t* dev = (ErrmemBackendBlockDevice_t*) backend;
	uint32_t old_msgmagic = getCurrentMsgMagic(dev->header->blockmagic);
	do {
		err = generate_block_magic(backend, ++it);
		if (!err)
			dev->msg_magic = getCurrentMsgMagic(dev->blockmagic);
	} while (!err && (old_msgmagic == dev->msg_magic));

	return err;
}

static char* get_command(ErrmemBackend_t* backend)
{
	if (backend)
		return ((ErrmemBackendBlockDevice_t*)backend)->command;
	return NULL;
}

static char* get_name(ErrmemBackend_t* backend)
{
	if (backend)
		return ((ErrmemBackendBlockDevice_t*)backend)->name;
	return NULL;
}

static unsigned get_nr(ErrmemBackend_t* backend)
{
	if(backend)
		return ((ErrmemBackendBlockDevice_t*)backend)->nr;
	return ~0;
}

static void set_nr(ErrmemBackend_t* backend, uint32_t pnr)
{
	if(backend)
		((ErrmemBackendBlockDevice_t*)backend)->nr = pnr;
	return;
}

static int32_t is_default(ErrmemBackend_t* backend)
{
	if(backend)
		return ((ErrmemBackendBlockDevice_t*)backend)->def;
	return 0;
}

static void set_default(ErrmemBackend_t* backend, int32_t val)
{
	if(backend)
		((ErrmemBackendBlockDevice_t*)backend)->def = val;
	return;
}

static unsigned get_type(ErrmemBackend_t* backend)
{
	if(backend)
		return ((ErrmemBackendBlockDevice_t*)backend)->type;
	return 0;
}

static unsigned get_last_seq_nr(ErrmemBackend_t* backend)
{
	if(backend)
		return ((ErrmemBackendBlockDevice_t*)backend)->last_stored;
	return 0;
}

// info - show information about this backend
static void info(struct ErrmemBackend* backend, int32_t fd)
{
	ErrmemBackendBlockDevice_t* dev = (ErrmemBackendBlockDevice_t*) backend;
	int32_t pos = 0;
	if (dev && dev->name) {
		if ((dev->status == STAT_OK) || (dev->status == STAT_ERASED)){
			pos = seek_dev(dev->fd, dev->name,  0, SEEK_CUR);
			if (pos < 0)
				syslog(LOG_CRIT, "%s %s", "failed to seek to current position SEEK_CUR - error: ", strerror(-pos));
			else {
				if (dev->blocksize)
					dprintf(fd,
							"Number: %d - Device: %s: type: %s  @block %u, blocksize %u, %u blocks, %u used blocks\n "
							"%u persistent blocks, alignment %u\n",
							dev->nr, dev->name, int_type2char_type(dev->type), ((uint32_t)pos / dev->blocksize), dev->blocksize, dev->blocks,
							dev->used, dev->persistent, dev->w_width);
				else
					syslog(LOG_CRIT, "%s", "blocksize is undefined when calculation values for info ");
			}
		} else
			dprintf(fd, "%s: block device - no information available\n", dev->name);
	}
}


// match - match an error memory message against a regular expression
static int32_t match(regex_t* filter, unsigned char* msg)
{
	return !filter || !regexec(filter, (char*) msg, 0, 0, 0);
}


struct filter
{
	regex_t*    regex;
	int32_t     first;
	int32_t     last;
	int32_t     n;
};

/* dump - Dump all stored messages into a given output backend.
 *        In client mode the session context has to be created.
 *        In server mode the session context is passed to this function.  
 */
static void dump(
	struct ErrmemBackend* backend, struct ErrmemBackend* out,
	struct SessionInfo* s, const char* filter, int32_t first, int32_t last)
{
	SessionInfo_t    si = {0};
	MessageQueue_t** mq = NULL;
	struct filter f = {0};
	regex_t compiled;

	if (out) {
		if (!s) {
			/* This is code not executed by the daemon rather than by application */
			if (backend) 
			{
				if (((ErrmemBackendBlockDevice_t*)(backend))->status == STAT_OK) {
					if (((ErrmemBackendBlockDevice_t*)(backend))->access_mode == O_RDONLY) {
						si.backend = backend;
						si.fd              = ((ErrmemBackendBlockDevice_t*)(si.backend))->fd;
						si.num             = ~0;
						si.num_start       = ~0;
						si.pos             =  0;
						si.seq_num         =  0;
						si.seq_max         = ((ErrmemBackendBlockDevice_t*)(si.backend))->blocknum;
						si.block_magic     = ~0;
						si.seq_last        = ~0;
						si.seq_next        = ~0;
						si.seq_new         = ~0;
						si.client_mode     =  1;
						si.persistent      =  0;
						si.msg_queue       = NULL;
						si.status          = ERR_OK;
						si.state           = ERRMEM_SESSION_ESTABLISHED;
						s = &si;
					} else
						syslog(LOG_CRIT, "%s %s %s",
							   "Server / Client mode mixed up. Device: ",
							   backend->get_name(backend), " will not be dumped");				
				}
			} else
				syslog(LOG_CRIT, "%s",
					   "No backend specified for application dump");				
		}

		if (s) {
			f.regex = NULL;
			if (filter && !regcomp(&compiled, filter, REG_NEWLINE))
				f.regex = &compiled;
			if (first > 0)
				f.first = first;
			if (last > 0)
				f.last = last;
			f.n = 0;
		
			do {
				if (!s->msg_queue) {
					s->msg_queue = calloc(1, sizeof(MessageQueue_t));
					if (s->msg_queue) {
						s->backend->read_session(s);
						errmem_create_messages(s);
					}
				}
				mq = errmem_get_message(NULL, &s->msg_queue);
				if (*mq) {
					if (match(f.regex,(*mq)->msg.message) &&
						(!f.last || (f.last && ++f.n <= f.last))) {
						if (out->store)
							out->store(out, offsetof(struct errmem_message, message) + 
									   (*mq)->msg.length, &((*mq)->msg));
					}
					if (mq && *mq) {
						free(*mq);
						*mq = NULL;
						mq  = NULL;
					}
				}
				if (f.last && f.n >= f.last)
					break;
			} while(s->state != ERRMEM_SESSION_READ_DONE &&
					s->state != ERRMEM_SESSION_BACKEND_ERASED &&
					s->status >= 0 && !terminate);
		}
	} else {
		syslog(LOG_CRIT, "%s", "No output backend to dump to specified");
	}
}

static int32_t set_erase_time(struct ErrmemBackend* backend, uint32_t start)
{
	int32_t rc = 0;
	ErrmemBackendBlockDevice_t* dev = (ErrmemBackendBlockDevice_t*) backend;
	struct timespec time_spec;
	if (dev && dev->wait && dev->times && dev->erasesize) {
		rc = clock_gettime(CLOCK_MONOTONIC_RAW, &time_spec);
		if (rc)
			rc = -errno;
		else
			dev->times[start / dev->erasesize] = time_spec.tv_sec;
	}
	return rc;
}

static int32_t check_and_wait_erase_cycle(struct ErrmemBackend* backend, uint32_t start)
{
	int32_t rc = 0;
	ErrmemBackendBlockDevice_t* dev = (ErrmemBackendBlockDevice_t*)backend;
	struct timespec time_spec;
	uint64_t elapsed = 0;
	int64_t  wait    = 0;
	if (dev && dev->wait && dev->times && dev->erasesize) {
		rc = clock_gettime(CLOCK_MONOTONIC_RAW, &time_spec);
		if (rc)
			rc = -errno;
		else {
		elapsed = time_spec.tv_sec - dev->times[start / dev->erasesize];
		wait = dev->wait - elapsed;
		if (wait > 0)
			sleep(wait);
		}
	}
	return rc;
}

static int32_t block_erase(struct ErrmemBackend* backend, int32_t pos)
{
	int32_t err = 0;
	ErrmemBackendBlockDevice_t* dev = (ErrmemBackendBlockDevice_t*) backend;
    erase_info_t e_info = { .start = pos, .length = dev->erasesize};
	uint32_t end   = pos + dev->blocksize;
	uint32_t magic = 0x00000000;

	if(dev && dev->name) {
		/* Invalidate Header */
		err = write_dev(dev->fd, dev->name, (char*)&magic, sizeof(uint32_t));
		if (err)
			syslog(LOG_CRIT, "%s %s %s", "Failed to invalidate NOR flash block ",
				   "with magic before erasing ", dev->name);
		if (err)
			/* don't overwrite the error already occured */
			(void)seek_dev(dev->fd, dev->name, pos, SEEK_SET);
		else {
			err = seek_dev(dev->fd, dev->name, pos, SEEK_SET);
			if (err < 0)
				syslog(LOG_CRIT, "%s %s %s", "Failed to seek to beginning of NOR flash block ",
					   "which shall be erased ", dev->name);
		}

		if (err >= 0) {
		/* Delete one block in chunks of ersasesize */
			for (; e_info.start < end; e_info.start += e_info.length) {
				err = check_and_wait_erase_cycle(backend, e_info.start);
				if (!err) {
					err = ioctl(dev->fd, MEMUNLOCK, &e_info);
					if (err && errno != EOPNOTSUPP) {
						err = -errno;
						syslog(LOG_CRIT, "%s %s %s %m", "Cannot unlock NOR flash device ",
							   dev->name, "for erasing eraseblock");
					} else {
						err = ioctl(dev->fd, MEMERASE, &e_info);
						if (err) {
							err = -errno;
							syslog(LOG_CRIT, "%s %d %s %d %s %s %m", "Failed to erase block from ",
								   e_info.start, " with erasesize ", e_info.length,
								   "in NOR flash device ", dev->name);
						}
					}
				} else 
					syslog(LOG_CRIT, "%s %s %m", "Failed to check erase cycle time for NOR flash device ",
						   dev->name);
				if (!err)
					set_erase_time(backend, e_info.start);

				if (err)
					break;
			}
		}
	} else {
		syslog(LOG_CRIT, "%s", "Invalid parameters in function errmem_backend_block_devcice::block_erase");
		err = -EINVAL;
	}
	return err;
}

static int32_t check_erase(struct ErrmemBackend* backend, int32_t pos, int32_t len)
{
	int32_t  err = 0;
	int32_t  ret = 0;
	uint32_t magic = 0;
	uint16_t crc   = 0;
	ErrmemBackendBlockDevice_t* dev = (ErrmemBackendBlockDevice_t*) backend;
	if (dev && dev->name) {
		/* save current position in storage */
		ret = seek_dev(dev->fd, dev->name, 0, SEEK_CUR);
		if (ret < 0)
			err = ret;
		else {
			/* seek to begin of current block */
			pos = seek_dev(dev->fd, dev->name, pos, SEEK_SET);
			if (pos < 0)
				err = pos;
			else {
				/* now we can read and check */
				void* check = malloc(len);
				if (!check)
					err = -ENOMEM;
				if (!err)
					err = read_dev(dev->fd, dev->name, check, len);
				if (err >= 0) {
					ErrmemBackendBlockDeviceHeader_t* h = NULL;
					ErrmemBackendBlockDeviceBlock_t*  b = NULL;
					PersistentMessage_t*              p = NULL;
					struct errmem_message*            m = NULL;
					/* check where we are and then go back to the car */
					magic = *(mkuint32(check, 0));
					if (magic == MAGIC_HEADER || magic == MAGIC_BLOCK_ZERO) {
						/* block zero */
						h = (ErrmemBackendBlockDeviceHeader_t*)((void*)check);
						if (h) {
							crc = h->crc;
							h->crc = 0;
							if (calc_crc16(0, (unsigned char*)h,
										   sizeof(ErrmemBackendBlockDeviceHeader_t))!= crc)
								err = -1;
							h->crc = crc;
							if (!err && (magic == dev->header->magic))
								b = (ErrmemBackendBlockDeviceBlock_t*)addptr((unsigned char*)check,
																			 aligned_len(dev->header->header_length, dev->w_width));
							if (!err && (magic == MAGIC_BLOCK_ZERO))
								b = NULL;
						} else
							err = -ENOMEM;
					} else if (magic == dev->header->blockmagic)
						b = (ErrmemBackendBlockDeviceBlock_t*)((void*)check);
					else
						err = -1;

					/* check block header */
					if (!err && b) {
						crc = b->crc;
						b->crc = 0;
						if (calc_crc16(dev->seed, (unsigned char*)b,
									   sizeof(ErrmemBackendBlockDeviceBlock_t))!= crc)
							err = -1;
						b->crc = crc;					
					}
					
					/* check message header */
					if (!err && b) {
						p = mkpmsg(&b->data[0], 0);
						if (p) {
							crc = p->crc;
							p->crc = 0;
							if (calc_crc16(dev->seed, (unsigned char*)p,
										   offsetof(PersistentMessage_t, msg)) != crc)
								err = -1;
							p->crc = crc;
						}
					}
					
					/* check message itself */
					if (!err && b && p) {
						m = mkmsg(&b->data[0], offsetof(PersistentMessage_t, msg));
						if (m) {
							crc = m->internal.crc;
							m->internal.crc = 0;
							if (calc_crc16(dev->seed, (unsigned char*)m, p->len) != crc)
								err = -1;
							m->internal.crc = crc;
						}
					}
				}
				if (check)
					free(check);
			}
			ret = seek_dev(dev->fd, dev->name, ret, SEEK_SET);
			if (ret < 0)
				err = ret;
		}
	} else
		err = -EINVAL;
	return err;
}

// erase - erase all stored messages
// err == 0 ==> ok
// err != 0 ==> error and backend will be destroyed
static int32_t erase(struct ErrmemBackend* backend)
{
	ErrmemBackendBlockDevice_t* dev = (ErrmemBackendBlockDevice_t*) backend;
	int32_t  err   = seek_dev(dev->fd, dev->name, 0, SEEK_SET);
    int32_t  rc    = 0;
	uint32_t al    = aligned_len(sizeof(ErrmemBackendBlockDeviceHeader_t), dev->w_width);
	int32_t  retry = 0;

	if (dev && dev->header && !err){
		if (dev->status != STAT_ABORT) {
			/* calculate new message magic and new blockmagic.
			 * Take care that neither blockmagic nor message magic
			 * will have the same old value.
			 */
			err = generate_new_magics((struct ErrmemBackend*)dev);
			if (err >= 0) {
				/* in case a client has to invalidate the content of a storage get a correct handle */
				if (dev->access_mode == O_RDONLY){
					close_dev(dev->fd, dev->name);
					dev->fd = -1;
					free(dev->header);
					dev->header = NULL;
					dev->fd = open_dev(dev->name, O_RDWR, 0);
					if (dev->fd >= 0) {
						dev->header = (ErrmemBackendBlockDeviceHeader_t*)get_mem(al);
						if (dev->header)
							memset(dev->header, 0xFF, al);
						else
							err = -ENOMEM;
					} else
						err = dev->fd;
				}

				if (err >= 0) {
					/* Destroy the magic of the header of the first block to indicate that
					 * the complete persistent storage is invalid.
					 * Store a new block magic to identify valid blocks. This makes it
					 * unnacessary to delete the block magics from all former valid blocks.
					 * Calculate a crc even for a header with a zero block magic in order
					 * to be able to use the other header values for checks during a new startup
					 */
					dev->header->magic         = MAGIC_BLOCK_ZERO;
					dev->header->crc           = 0;
					dev->header->header_length = sizeof(ErrmemBackendBlockDeviceHeader_t);
					dev->header->blocksize     = dev->blocksize;
					dev->header->persistent    = dev->persistent;
					dev->header->blocks        = dev->blocks;
					dev->header->blockmagic    = dev->blockmagic;
					dev->header->w_width       = dev->w_width;
					dev->header->crc = calc_crc16(0, (unsigned char*)dev->header,
												  sizeof(ErrmemBackendBlockDeviceHeader_t));	

					if (dev->type == BLOCK_NOR_RAW)
						retry = dev->retry;

					do {
						if (dev->type == BLOCK_NOR_RAW) {
							err = block_erase(backend, 0);
							/* guarantee the that file handle points to pos when returning
							 * from this function */
							if (err >= 0) {
								err = seek_dev(dev->fd, dev->name, 0, SEEK_SET);
							}
						}
						if (err >= 0) {
							err = write_dev(dev->fd, dev->name, (char*)dev->header, al);
						}

						if ((err >= 0) && (dev->type == BLOCK_NOR_RAW))
							err = check_erase(backend, 0, al);

						if ((err < 0) && (dev->type == BLOCK_NOR_RAW)) {
							rc = seek_dev(dev->fd, dev->name, 0, SEEK_SET);
							if (rc < 0) {
								dev->status = STAT_ABORT;
								err = rc;
							}
						}
						retry--;
					} while ((err < 0) && (dev->type == BLOCK_NOR_RAW) && (dev->status != STAT_ABORT) && (retry >= 0));
						
					if (dev->access_mode == O_RDONLY){
						err = close_dev(dev->fd, dev->name);
						dev->fd = -1;
						if (err >= 0) {
							dev->fd = open_dev(dev->name, O_RDONLY, 0);
							err = dev->fd;
						}
					}					

					if (err < 0) 
						dev->status = STAT_ABORT;
					else {
						//reset block device values
						dev->status            = STAT_ERASED;
						dev->blocknum          = 0;
						dev->seqnum            = ~0;
						dev->flags             = 0;
						dev->storage           = 0;
						dev->writepos          = 0;
						dev->seed              = 0;
						dev->used              = 0;
						dev->last_stored       = 0;
						//reset 0-block magic
						//Note: dev->block->magic refers to BlockHeaderMagic not to block magic because of 0-block
						//When creating a new 0-block indicate that this is the zero block to inform store function
						//to consider the device block header
						dev->block->magic = dev->header->magic;
						err = seek_dev(dev->fd, dev->name, 0, SEEK_SET);

						if (err < 0)
							dev->status = STAT_ABORT;
					}
				} else
					syslog(LOG_INFO, "%s %s", "device is not erased - couldn't get file handle or memory ", dev->name);
			} else
				syslog(LOG_INFO, "%s %s", "device is not erased - new block magic couldn't be generated ", dev->name);
		} else {
			syslog(LOG_INFO, "%s %s", "device is set to state aborted - perhaps already erased or corrupted ", dev->name);
			err = -ENOEXEC;
		}
	} else {
		syslog(LOG_INFO, "%s ", "device is set to state aborted - device access or parameter check failed");
		err = -EINVAL;
	}
	return err;
}

static uint32_t prepare_new_block(struct ErrmemBackend* backend, PersistentMessage_t* pmsg)
{
	uint32_t  rc                           = ~0;
	int32_t   err                          = 0;
	uint32_t  offset                       = 0;
	uint32_t  entry_len                    = 0;
	ErrmemBackendBlockDevice_t* dev        = (ErrmemBackendBlockDevice_t*) backend;
	int32_t pos                            = seek_dev(dev->fd, dev->name,  0, SEEK_CUR);
	ErrmemBackendBlockDeviceBlock_t* block = dev->block;

	/* store the current block magic (block or header magic) */
	uint32_t tmp                           = dev->block->magic;
	uint32_t flags                         = pmsg->msg->internal.flags;
	int32_t  retry                         = 0;
	
	if (pos >= 0) {
		/* pos is inside the last block and we need a new one ==> wrap around */
		if ((uint32_t)pos > (dev->blocks -1) * dev->blocksize) {
			// wrap around to the begin of the first block that can be overwritten
			pos = seek_dev(dev->fd, dev->name, dev->persistent * dev->blocksize, SEEK_SET);
			if (!pos) {
				/* it is block 0 (pos == 0) after a wrap around ==> increase blocknum 
				 * Note: do not wirte to dev->block here. It will be initialized later to FF */
				++dev->blocknum;
				tmp = MAGIC_BLOCK_ZERO;
			}
		} else if (dev->block->magic == MAGIC_BLOCK_ZERO)  {
			// we start from the beginning and we are completely empty
			pos = seek_dev(dev->fd, dev->name, 0, SEEK_SET);
        } else {
			// get the next block
			if ((uint32_t)pos % dev->blocksize != 0)
				pos = seek_dev(dev->fd, dev->name, (pos - ((uint32_t)pos % dev->blocksize)) + dev->blocksize, SEEK_SET);
        }
	}

	/* error when seeking on the device ==> abort because cursor position is undefined */
	if (pos < 0) {
		dev->status = STAT_ABORT;
		err = pos;
	}

	if (dev->status == STAT_OK || dev->status == STAT_ERASED) {
		// init block with 0xFF and set blocknum. We are starting with blocknumber 0 
		memset(block, 0xFF, dev->blocksize);

		if (tmp != MAGIC_BLOCK_ZERO)
			/* it is not block zero ==> increase blocknum here */
			++dev->blocknum;
		else {
			/* it is block zero - don't increase blocknum here. If it is no wrap around
			 * we want to start with block number 0
			 */
			dev->header->magic         = MAGIC_HEADER;
			dev->header->header_length = sizeof(ErrmemBackendBlockDeviceHeader_t);
			dev->header->crc           = 0;
			dev->header->blocksize     = dev->blocksize;
			dev->header->persistent    = dev->persistent;
			dev->header->blocks        = dev->blocks;
			dev->header->blockmagic    = dev->blockmagic;
			/* write block device header information to block structure */
			*(ErrmemBackendBlockDeviceHeader_t*)((void*)dev->block) = *dev->header;
			/* the checksum for the block device header will be the seed for calculation
			 * of the crc of block headers.
			 */
			dev->seed = calc_crc16(0, (void*)dev->block, sizeof(ErrmemBackendBlockDeviceHeader_t));
			((ErrmemBackendBlockDeviceHeader_t*)((void*)dev->block))->crc = dev->seed;
			dev->header->crc = dev->seed;

			/* consider the block device header (block zero) and align the address */
			offset = aligned_len(dev->header->header_length, dev->w_width);
			block = (ErrmemBackendBlockDeviceBlock_t*)addptr((unsigned char*)dev->block, offset);
		}
		
		/* initialize the block header */
		block->magic    = dev->blockmagic;
		block->reserved = ~0;
		block->blocknum = dev->blocknum;

		pmsg->blocknum = dev->blocknum;

		// check entry length to fit in the block
		entry_len = aligned_len(offset + ((sizeof(ErrmemBackendBlockDeviceBlock_t) - 1) +
										  offsetof(PersistentMessage_t, msg) + pmsg->len), dev->w_width);
		if (entry_len > dev->blocksize) {
			pmsg->len = pmsg->len - (entry_len - dev->blocksize);
			pmsg->msg->internal.flags |= ERRMEM_FLAG_INCOMPLETE;
			entry_len = offset + aligned_len((sizeof(ErrmemBackendBlockDeviceBlock_t) + 
									 offsetof(PersistentMessage_t, msg) + pmsg->len), dev->w_width);
		}

		/* Calculate crc for the header information of a persistent message.
		 * No need to initialize psmg.crc with zero. That's done in function store when
		 * creating the pmsg struct on the stack.
		 */
		pmsg->crc = calc_crc16(dev->seed, (void*)pmsg, offsetof(PersistentMessage_t, msg));
		/* write magic and length */
		memcpy(addptr((unsigned char*)&block->data, 0), pmsg, offsetof(PersistentMessage_t, msg));

		pmsg->msg->internal.flags &=
			~(dev->flags & (ERRMEM_FLAG_RESTART |
							ERRMEM_FLAG_INITIALIZED |
							ERRMEM_FLAG_DROPPED));
	
		// calculate the message internal checksum and set the overall message number
		pmsg->msg->internal.storage = ++dev->storage;
		pmsg->msg->internal.crc = 0;
		pmsg->msg->internal.crc = calc_crc16(dev->seed, (void*) pmsg->msg, pmsg->len);
		// write the message to the block structure
		memcpy(addptr((unsigned char*)&block->data, offsetof(PersistentMessage_t, msg)), pmsg->msg, pmsg->len);
	
		// now calculate the checksum for the block
		block->crc = 0;
		block->crc = calc_crc16(dev->seed, (void*) block,
								sizeof(ErrmemBackendBlockDeviceBlock_t));

		if (dev->type == BLOCK_NOR_RAW)
			retry = dev->retry;
		// here we go for special handling of bare NOR flash devices
		do {
			if (dev->type == BLOCK_NOR_RAW) {
				err = block_erase(backend, pos);
				/* guarantee the that file handle points to pos when returning
				 * from this function */
				if (err >= 0)
					err = seek_dev(dev->fd, dev->name, pos, SEEK_SET);
			}
			// here write the complete entry for a new block
			if (err >= 0)
				err = write_dev(dev->fd, dev->name, (void*)dev->block, entry_len);

			if ((err >= 0) && (dev->type == BLOCK_NOR_RAW))
				err = check_erase(backend, pos, entry_len);

			if ((err < 0) && (dev->type == BLOCK_NOR_RAW)) {
				pos = seek_dev(dev->fd, dev->name, pos, SEEK_SET);
				if (pos < 0) {
					dev->status = STAT_ABORT;
					err = pos;
				}
			}
			retry--;
		} while ((err < 0) && (dev->type == BLOCK_NOR_RAW) && (dev->status != STAT_ABORT) && (retry >= 0));

		if (err >= 0) {
			dev->writepos = (entry_len - (offset + sizeof(ErrmemBackendBlockDeviceBlock_t))) + 1;
			dev->status   = STAT_OK;
			// set the current position of the file cursor behind the message written (position related to
			// start of the persistent memory)
		} else
			dev->status = STAT_ABORT;
	}
	/* return of store function is the sequence number or -1 in case of error */
	if (err >= 0)
		rc = pmsg->msg->internal.seqnum;

	dev->flags = flags;
	dev->seqnum = ~0;
	return rc;
}

// store - store an error memory message
// Return: err >= 0 ==> internal sequence number of current message
//         err = -1 ==> error ==> Abort
static uint32_t store(
	struct ErrmemBackend* backend, int len,
	struct errmem_message* msg)
{
	ErrmemBackendBlockDevice_t* dev        = (ErrmemBackendBlockDevice_t*) backend;
	ErrmemBackendBlockDeviceBlock_t* block = NULL;
	PersistentMessage_t pmsg               = {.magic = dev->msg_magic, .crc = 0,
											  .reserved = ~0, .blocknum = 0,
											  .len = len, .msg = msg};
	uint32_t seqnum                        = 0;
	uint32_t offset                        = 0;
	uint32_t flags                         = msg->internal.flags;
	int32_t remain                         = 0;
	int32_t err                            = 0;

	if (dev && (dev->status == STAT_OK || dev->status == STAT_ERASED)) {
		/* if this message is already written skip it */
		if (flags & ERRMEMD_SEQ_NR_TWICE) {
			seqnum = msg->internal.seqnum;
			dev->flags = flags;
			dev->seqnum = seqnum;
		} else {
			if ((dev->block->magic == MAGIC_BLOCK_ZERO) || dev->block->magic == MAGIC_HEADER)
				offset = aligned_len(dev->header->header_length, dev->w_width);

			remain = dev->blocksize - (offset + 
						   (sizeof(ErrmemBackendBlockDeviceBlock_t) - 1) +
						   dev->writepos);
	
			/* store the act of ignoring of the last message with this subsequent message */
			if (dev->flags & ERRMEMD_SEQ_NR_TWICE) {
				msg->internal.flags |= ERRMEMD_SEQ_NR_TWICE;
				dev->flags &= ~ERRMEMD_SEQ_NR_TWICE;
				if (!msg->internal.offset)
					msg->internal.offset = dev->seqnum;
				dev->seqnum = ~0;
			}
			// If a new block is needed handle it here. This is because the new message
			// will not fit into the current block or the current block is the unused 0-Block
			if ( remain <= 0 ||
				 (uint32_t)remain < aligned_len((offsetof(PersistentMessage_t, msg) + pmsg.len), dev->w_width) ||
				 ( (dev->block->magic != dev->blockmagic) && (dev->block->magic != MAGIC_HEADER)))  {
				seqnum = prepare_new_block(backend, &pmsg);
			} else {
				block = (ErrmemBackendBlockDeviceBlock_t*)addptr((unsigned char*)dev->block, offset);
				pmsg.blocknum = block->blocknum;
				pmsg.crc = calc_crc16(dev->seed, (void*)&pmsg, offsetof(PersistentMessage_t, msg));
				memcpy(&block->data[dev->writepos], &pmsg, offsetof(PersistentMessage_t, msg));
		
				// delete some flags in successive messages
				msg->internal.flags &=
					~(dev->flags & (ERRMEM_FLAG_RESTART |
									ERRMEM_FLAG_INITIALIZED |
									ERRMEM_FLAG_DROPPED));

				// calculate the crc
				msg->internal.storage = ++dev->storage;
				msg->internal.crc = 0;
				msg->internal.crc = calc_crc16(dev->seed, (void*) pmsg.msg, pmsg.len);
				memcpy(&block->data[dev->writepos + offsetof(PersistentMessage_t, msg)], pmsg.msg, pmsg.len);
			
				err = write_dev(dev->fd, dev->name,
								(void*)&block->data[dev->writepos],
								aligned_len((offsetof(PersistentMessage_t, msg) +
											 pmsg.len), dev->w_width));
				if (err < 0) {
					dev->status = STAT_ABORT;
					seqnum = ~0;
				} else {
					dev->writepos += aligned_len((offsetof(PersistentMessage_t, msg) + pmsg.len), dev->w_width);
					seqnum = msg->internal.seqnum;
					dev->flags = flags;
					dev->seqnum = ~0;
				}
			}
		}
	} else
		seqnum = ~0;
	return seqnum;
}

// destroy - free resources
static void destroy(struct ErrmemBackend* backend)
{
	ErrmemBackendBlockDevice_t* dev = (ErrmemBackendBlockDevice_t*) backend;

	if (dev) {
		if (dev->fd >= 0)
			close(dev->fd);

		if (dev->name)
			free(dev->name);

		if (dev->command)
			free(dev->command);

		if (dev->header)
			free(dev->header);

		if (dev->block)
			free(dev->block);
	
		if (dev->times)
			free(dev->times);

		free(dev);
	}
}

static int32_t backup(ErrmemBackendBlockDevice_t* dev, int32_t action)
{
    int32_t  err    = 0;
	int32_t  sz     = 0;
	int32_t  dst_fd = -1; 
	char    *ndest  = NULL;
	struct stat buf = {0};

	if (dev && dev->name) {
		sz = strlen(dev->name) + 4 /* .bck */ + 1 /* \0 */;
		ndest = (char*)calloc(1, sz);
		if (ndest) {
			if (snprintf(ndest, sz,"%s.bck", dev->name) >= sz) {
				err = -ENAMETOOLONG;
				syslog(LOG_CRIT, "%s %s %s %s",
					   "Not enough memory to create backup destination file ",
					   dev->name, ".bck - error = ", strerror(-err));
			} else if (action & BACKEND_COPY) {
				dst_fd = open_dev(ndest, O_WRONLY | O_CREAT, 0);
				if (dst_fd >= 0) {
					err = stat(dev->name, &buf);
					if (-1 == err) {
						err = -errno;
						syslog(LOG_CRIT, "%s %s %s %s", "Stat failed on device "
							   "for backup ", dev->name, " - error = ",
							   strerror(-err));
					} else {
						err = sendfile(dst_fd, dev->fd, 0, buf.st_size);
						if (err < 0) {
							err = -errno;
							syslog(LOG_CRIT, "%s %s %s %s",
								   "Cannot copy device ", dev->name,
								   " - error = ", strerror(-err));
						} else {
							/* If the original file is larger in size than the 
							 * currently requested size, truncate the filesize
							 * accordingly. Don't check here if really 
							 * neccessary to avoid unneccessary overhead. 
							 */
							err = ftruncate(dev->fd, 
											(dev->blocksize * dev->blocks));
							if (err < 0) {
								err = -errno;
								syslog(LOG_CRIT, "%s %s %s %s",
									   "Failed to truncate file ", dev->name,
									   " - error = ", strerror(-err));
							}
						}
					}
					err = close_dev(dst_fd, ndest);
				} else {
					err = -errno;
					syslog(LOG_CRIT, "%s %s %s %s",
						   "Cannot open destination device ", ndest,
						   " - error = ", strerror(-err));
				}
			} else if (action & BACKEND_MOVE) {
				if (dev->fd >= 0) {
					err = close_dev(dev->fd, ndest);
					dev->fd = -1;
				}
				if (err >= 0) {
					err = rename(dev->name, ndest);
					if (err < 0) {
						err = -errno;
						syslog(LOG_CRIT, "%s %s %s %s",
							   "Can't move to destination device ", ndest,
							   " - error = ", strerror(-err));
					}
				}
			} else {
				err = -EBADRQC;
				syslog(LOG_CRIT, "%s %s",
					   "Wrong action code in backup function"
					   " - error = ", strerror(-err));
			}
			free(ndest);
		} else {
			err = -ENOMEM;
			syslog(LOG_CRIT, "%s %s %s %s",
				   "No memory for name of backup file ",
				    dev->name, ".bck - error = ", strerror(-err));
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s",
			   "backup: parameter check failed - error = ", strerror(-err));
	}

	return err;
}

static int32_t create_dev(ErrmemBackendBlockDevice_t* dev)
{
    int32_t err = -1;
	if (dev) {
		if (dev->fd >= 0) {
			(void)close_dev(dev->fd, dev->name);
			dev->fd = -1;
		}
		dev->fd = open_dev(dev->name, (dev->access_mode | O_CREAT), 0);
		if (dev->fd >= 0) {
			err = ftruncate(dev->fd, (off_t)(dev->blocksize * dev->blocks));
			if (err < 0) {
				err = -errno;
				syslog(LOG_CRIT, "%s %ld %s %s",
					   "create_dev: cannot truncate file to size",
					   (off_t)(dev->blocksize * dev->blocks), " error = ",
					   strerror(-err));
			}
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s",
			   "create_dev: parameter check failed error = ", strerror(-err));
	}

    return err;
}

static int32_t check_parms(ErrmemBackendBlockDevice_t *dev)
{
	int32_t err = 0;
	if (!dev || !dev->name) {
		err = -EINVAL;
	    syslog(LOG_CRIT, "%s", "check_parms (block device backend): parameter dev invalid");
	} else {
		if (dev->access_mode == O_RDONLY) {
			if (dev->blocksize)
				dev->blocksize = 0;
			if (dev->blocks)
				dev->blocks = 0;
			if (~dev->persistent)
				dev->persistent = ~0;
			if (~dev->w_width)
				dev->w_width = ~0;
		} else if (dev->access_mode & O_RDWR) {
			/* Accessing a persistent storage  in RDWR mode
			 * requires the definition of at least blocksize, 
			 * blockcount and number of persistent blocks. */
			if (!dev->blocksize || !~dev->blocksize) {
				err = -EINVAL;
				syslog(LOG_CRIT, "%s %s", "check_parms (block device backend): "
					   "blocksize undefined", dev->name);
			} else if (dev->blocksize < STD_SECTOR_SIZE) {
				err = -EINVAL;
				syslog(LOG_CRIT, "%s %d %s %s", "check_parms (block device backend): "
					   "blocksize ", dev->blocksize, " less than 512", dev->name);
			} else {
				uint32_t v = dev->blocksize;
				/* Every single bit starting at position bit 9 represents a valid
				 * block size.
				 * 31 ---------- bit number ------------- 0
				 *  0000 0000 0000 0000 0000 0010 0000 0000 :  512
				 *  0000 0000 0000 0000 0000 0100 0000 0000 : 1024
				 *               ....
				 *  0100 0000 0000 0000 0000 0000 0000 0000 : (not really of interest)
				 *  1000 0000 0000 0000 0000 0000 0000 0000 : (not really of interest)
				 * ==> mask represents the value having every bit set starting at
				 *     position 9.
				 * mask: 1111 1111 1111 1111 1111 1110 0000 0000
				 */
				uint32_t mask = (uint32_t)(((uint64_t)1 << 32) - 512);
				while (v & mask) {
					/* v is greater than 512 */
					/* There is no remaining rest less than 512 */
					if(!(v & ~mask))
						v >>=1;
					else {
						err = -EINVAL;
						syslog(LOG_CRIT, "%s %d %s",  "chek_parms (block device backend): "
							   "Blocksize ", dev->blocksize, " is invalid. Blocksize has to "
							   "have a value of 512 * 2**(n-1)");
						break;
					}
				}
			}
			if (!dev->blocks || !~dev->blocks) {
				err = -EINVAL;
				syslog(LOG_CRIT, "%s %s",  "check_parms (block device backend): "
					   "blockcount undefined", dev->name);
			}
			if (!~dev->persistent) {
				err = -EINVAL;
				syslog(LOG_CRIT, "%s %s", "check_parms (block device backend): "
					   "persist blocks undefined", dev->name);
			}
			if (dev->type == BLOCK || dev->type == BLOCK_RAW || dev->type == BLOCK_FLASH) {
				if ((dev->w_width < DEFAULT_ALIGNMENT) || !~dev->w_width)
					dev->w_width = DEFAULT_ALIGNMENT;
			} else if (dev->type == BLOCK_FLASH_RAW) {
				if ((dev->w_width < DEFAULT_ALIGNMENT) || (!~dev->w_width)) {
					err = -EINVAL;
					syslog(LOG_CRIT, "%s %s %s", "check_parms (block device backend): "
						   "write size to use for raw flash access undefined - device: ",
						   dev->name ," will be ignored!");
				}
			} else {
				err = -EINVAL;
				syslog(LOG_CRIT, "%s %s %s", "check_parms (block device backend): "
					   "device type undefined - device: ", dev->name, " will be ignored!");
			}
			if ((0 < dev->w_width) && (0 < dev->blocksize) &&
				(aligned_len(sizeof(ErrmemBackendBlockDeviceHeader_t), dev->w_width) +
				 aligned_len((sizeof(ErrmemBackendBlockDeviceBlock_t) - 1) +
							 offsetof(PersistentMessage_t, msg) +
							 sizeof(struct errmem_message), dev->w_width)) >
				dev->blocksize) {
				err = -EINVAL;
				syslog(LOG_CRIT, "%s %s", "check_parms (block device backend): "
					   "write alignment does not allow to store at least one message in a block - device: ",
					   dev->name);
			}
			if ((~dev->persistent) && (dev->persistent)
				&& (dev->persistent >= dev->blocks)) {
				err = -EINVAL;
				syslog(LOG_CRIT, "%s %s", "check_parms (block device backend): "
					   "Persistent area is equal or bigger than storage - device: ",
					   dev->name);
			}
			if ((~dev->w_width) && (dev->w_width) && (dev->w_width > dev->blocksize)) {
				err = -EINVAL;
				syslog(LOG_CRIT, "%s %s", "check_parms (block device backend): "
					   "write size bigger than storage blocksize - device: ",
					   dev->name);
			}
		} else {
			err = -EINVAL;
			syslog(LOG_CRIT, "%s %s", "check_parms (block device backend): "
				   "access mode undefined - device: ", dev->name);
		}
	}
	return err;
}

static int32_t validate_dev_header(ErrmemBackendBlockDevice_t* dev)
{
	/* check the crc */
	int32_t  err = 0;
	uint16_t crc = 0;
	
	if (dev && dev->header && dev->name) {
		crc = dev->header->crc;
		dev->header->crc = 0;
		if (calc_crc16(0, (unsigned char*)dev->header,
					   sizeof(ErrmemBackendBlockDeviceHeader_t))!= crc)
			err = -1;
		dev->header->crc = crc;
	}
	return err;
}

static int32_t check_dev_header(ErrmemBackendBlockDevice_t* dev)
{
	/* check the crc to get an idea whether values from header 
	 * can be used or not */
	int32_t  err = 0;
	if (dev && dev->header && dev->name) {
		err = validate_dev_header(dev);
		/* ToDo: reset error in order to care for device correctly */
		if (!err) {
			if (dev->header->header_length != 
				sizeof(ErrmemBackendBlockDeviceHeader_t)) {
				if (dev->access_mode == O_RDONLY) {
					syslog(LOG_INFO, "%s %s", "check_dev_header (block device backend): "
						   "check version of error memory backend - expected header length "
						   "does not match - client aborts", dev->name);
					err = -EMEDIUMTYPE;
					dev->status = STAT_ABORT;
				} else if ((dev->type == BLOCK) &&
						   (dev->access_options & OPT_ACC_CREATE))
					dev->status = STAT_EX_CREATE;					
				else
					dev->status = STAT_EX_ERASE;
			}
			if (!err && dev->status == STAT_OK) {
				if (dev->access_mode == O_RDONLY) {
					if (dev->header->magic == MAGIC_HEADER) {
						dev->status = STAT_OK;
					} else if (dev->header->magic == MAGIC_BLOCK_ZERO){
						dev->status = STAT_ERASED;
						syslog(LOG_INFO, "%s %s", "check_dev_header (block device backend): "
							   "device is marked as erased - client exits", dev->name);
					} else {
						dev->status = STAT_ABORT;
						syslog(LOG_INFO, "%s %s", "check_dev_header (block device backend): "
							   "device is not initialized or may be corrupted - client exits", dev->name);
						err = -EIO;
					}
					if (dev->status != STAT_ABORT){
						dev->blocksize  = dev->header->blocksize;
						dev->blocks     = dev->header->blocks;
						dev->persistent = dev->header->persistent;
						dev->w_width    = dev->header->w_width;
					}
				} else {
					/* check haeder information against the command line values */
					if (dev->blocksize != dev->header->blocksize ||
						dev->blocks    != dev->header->blocks    ||
						dev->persistent!= dev->header->persistent||
						dev->w_width   != dev->header->w_width) {
						if ((dev->type == BLOCK) &&
						   (dev->access_options & OPT_ACC_CREATE))
							dev->status = STAT_EX_CREATE;
						else
							dev->status = STAT_EX_ERASE;

						dev->header->blocksize  = 0;
						dev->header->blocks     = 0;
						dev->header->persistent = 0;
						dev->header->w_width    = 0;

					} else if (dev->header->magic == MAGIC_HEADER)
						dev->status = STAT_OK;
					else if (dev->header->magic == MAGIC_BLOCK_ZERO)
						dev->status = STAT_ERASED;
					else {
						if ((dev->type == BLOCK) &&
						   (dev->access_options & OPT_ACC_CREATE))
							dev->status = STAT_EX_CREATE;
						else
							dev->status = STAT_EX_ERASE;
					}
				}
			}
		} else {
			/* error already set here */
			if (dev->access_mode == O_RDONLY) {
				dev->status = STAT_ABORT;
				syslog(LOG_INFO, "%s %s", "check_dev_header (block device backend): "
						"header validation failed - Client exits: ", dev->name);
			} else {
				if ((dev->type == BLOCK) &&
					(dev->access_options & OPT_ACC_CREATE))
					dev->status = STAT_EX_CREATE;
				else
					dev->status = STAT_EX_ERASE;
				err = 0;
			}
		}
	} else {
		syslog(LOG_CRIT, "%s", "check_dev_header (block device backend): "
				"parameter check for dev failed");
		err = -EIO;
	}
	return err;
}

static int32_t setup_dev(ErrmemBackendBlockDevice_t* dev)
{	
	int32_t err = 0;
	if (!dev) {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s", "setup_dev: parameter check for dev failed");
	} else if (dev->status == STAT_ABORT)
		err = -ECANCELED;
	else if	(dev->status == STAT_EX_CREATE) {
		if (dev->type != BLOCK) {
			syslog(LOG_CRIT, "%s %s",
				   "setup_dev: Cannot create a raw block or a flash device ",
				   dev->name);
			err = -EACCES;
		} else {
			if (dev->access_options & OPT_ACC_BACKUP)
				err = backup(dev, BACKEND_MOVE);
			if (err >= 0) {
				err = create_dev(dev);
				if (err >= 0) {
					err = erase((ErrmemBackend_t*)dev);
				}
			}
		}
	} else if (dev->status == STAT_EX_ERASE) {
		if ((dev->type == BLOCK) && (dev->access_options & OPT_ACC_BACKUP))
			err = backup(dev, BACKEND_COPY);
		if (err >= 0)
			err = erase((ErrmemBackend_t*)dev);
	} else if (dev->status == STAT_CREATE) {
		if (dev->type != BLOCK) {
			syslog(LOG_CRIT, "%s %s",
				   "setup_dev: Cannot create a raw block or flash device",
				   dev->name);
			err = -EACCES;
		} else {
			err = create_dev(dev);
			if (err >= 0)
				err = erase((ErrmemBackend_t*)dev);
		}
	}
	if ((err < 0) && dev
		&& (dev->status != STAT_OK) && (dev->status != STAT_ERASED)) {
		err = -ECANCELED;
		syslog(LOG_CRIT,
			   "%s %s %s %s", "setup_dev: Device: ", dev->name,
			   " cannot be handled: error = ", strerror(-err));
	}
	return err;
}

static int32_t check_system_view(ErrmemBackendBlockDevice_t* dev)
{
	int32_t err  = 0;
	uint32_t sz  = 0;
	if (dev && dev->header && dev->name && dev->blocksize) {
		/* In case the device is already marked as to be created or to
		 * stop processing ommit system view check.
		 */
		if (dev->status != STAT_ABORT && dev->status != STAT_EX_CREATE &&
			dev->status != STAT_CREATE) {
			/* In the last step check the system view. Normally a file
			 * system should always know the size of its files */
			if (dev->stat_size)
				sz = dev->stat_size;
			else if ((dev->type == BLOCK_RAW) && dev->stat_blocks)
				sz = dev->stat_blocks * STD_SECTOR_SIZE;
			else {
				sz = 0;
				syslog(LOG_CRIT, "%s %s %s", "check_system_view (block_device_backend): "
					   "Cannot determine size of existing device: ", dev->name,
					   "Device will be ignored.");
				err = -EINVAL;
			}
			if (!err) {
				if (dev->type == BLOCK) {
					if (dev->access_mode == O_RDONLY) {
						/* Read only the number of available blocks. We should have
						 * valid header information here. So take dev->blocksize as the correct
						 * blocksize
						 */ 
						if (sz < (dev->blocksize * dev->blocks)) {
							/* system size information is less then information from  
							 * from device meta data.
							 */
							dev->blocks = sz / dev->blocksize;
							syslog(LOG_INFO,"%s %s %s %d %s %d %s %d %s %d %s %u", 
								   "check_system_view (block device backend): "
								   "dev: ", dev->name, " is of different size - values set to: "
								   "blocksize: ", dev->blocksize, " blocks: ", dev->blocks, "\n - system values are: "
								   "stat_blocksize: ",dev->stat_blocksize, " stat_blocks: ", dev->stat_blocks, " real size: ",
								   sz);
						}
					} else {
						/* We have to pay attention here regarding create options!
						 * In case create option is not given but given command line
						 * values cannot be satisfied with the existing device size
						 * we have to abort here. In case we are not allowed to create
						 * the device, but the existing device is big enough we can 
						 * re-initialize the device with given size. In
						 * case we are allowed to create the device  we will recreate
						 * the device.
						 * In any good cases keep in mind what the checks done before
						 * have shown.
						 */
						if (sz < (dev->blocksize * dev->blocks)) {
							if (dev->access_options & OPT_ACC_CREATE)
								dev->status = STAT_EX_CREATE;
							else {
								dev->status = STAT_ABORT;
								err = -EFBIG;
								syslog(LOG_INFO," %s %d %s %d %s %s", "check_system_view (block device backend): "
									   "Size of existing device is less than size specified. Recreation not "
									   "allowed. Creation option not set.\n Stat size = ", dev->stat_size,
									   " Specified size = ", (dev->blocksize * dev->blocks), " Device: ", dev->name);
							}
						} else if (sz > (dev->blocksize * dev->blocks)) {
							if (dev->access_options & OPT_ACC_CREATE)
								dev->status = STAT_EX_CREATE;
							else
								dev->status = STAT_EX_ERASE;
						}
					}
				} else { /* check for all other access beside filesystem on block device */
					if ((dev->blocksize * dev->blocks) > sz) {
						if (dev->access_mode == O_RDONLY) {
							dev->blocks = sz / dev->blocksize;
							syslog(LOG_INFO,"%s %s %s %d %s %d %s %d %s %d %s %u", 
								   "check_system_view (block device backend): "
								   "dev: ", dev->name, " is of different size - values set to: "
								   "blocksize: ", dev->blocksize, " blocks: ", dev->blocks, "\n - system values are: "
								   "stat_blocksize: ",dev->stat_blocksize, " stat_blocks: ", dev->stat_blocks, " real size: ",
								   sz);
						} else {
							dev->status = STAT_ABORT;
							err = -EFBIG;
							syslog(LOG_CRIT, "%s %s %s %d %s %d %s %u",
								   "check_system_view (block device backend): "
								   "dev: ", dev->name, " too big - values are "
								   "blocksize: ", dev->blocksize, " blocks: ", dev->blocks, "real size: ",
								   sz);
							}
						}
					}
				}
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s", "check_system_view: parameter check for dev failed");
	}
	return err;
}
// errmem_backend_create_block_device
// - create a block device error memory backend
static int32_t create(int32_t acc, struct ErrmemBackend* next)
{
    int32_t  err                    = 0;
	ErrmemBackendBlockDevice_t* dev = (ErrmemBackendBlockDevice_t*)next;

	if (dev && dev->name) {
		dev->access_mode = acc;
		/* check whether the device is accessible. Only in case of a file on a 
		 * block device which shall be handled blockwise and which is accessed
		 * using a file system, it is tried to create that file if it does not
		 * exist */
		err = stat_dev(dev);
		if (err >= 0) {
			dev->fd = open_dev(dev->name, dev->access_mode, 0);
			if (dev->fd < 0) {
				err = dev->fd;
				dev->fd = -1;
				dev->status = STAT_ABORT;
			} else
				/* If this advice fails, continue */
				(void)posix_fadvise(dev->fd, 0, 0, 
									POSIX_FADV_RANDOM | POSIX_FADV_NOREUSE);
		} else {
			/* the device requested does not exist */
			if (dev->access_mode != O_RDONLY) {
				/* for write access the device access mode has to be provided 
				 * as command line parameter */
				if ((dev->type == BLOCK) && (err == -ENOENT)) {
					/* requested device is a non existing file device on a
					 * block medium which is not flash. We will try to create it.*/
					if (dev->access_options & OPT_ACC_CREATE) {
						dev->status = STAT_CREATE;
						err         = 0;
					} else {
						syslog(LOG_CRIT, " %s %s %s ",
							   "create (block device backend): "
							   "Device ", dev->name, 
							   ": File on block device does not exist! "
							   "Access options does not allow creation."
							   " - It will be ignored ");
						dev->status = STAT_ABORT;
					}
				} else {
					syslog(LOG_CRIT, " %s %s %s %s %s %s",
						   "create block device backend: ",
						   int_type2char_type(dev->type), "- Device ",
						   dev->name,
						   ": either not exists or has conflicting device type"
						   " - error = ",
						   strerror(-err));
					dev->status = STAT_ABORT;
				}
			} else {
				syslog(LOG_CRIT, " %s %s %s %s %s %s",
					   "create block device backend: ",
					   int_type2char_type(dev->type), "- Device ", dev->name,
					   ": either not exists or has conflicting device type"
					   " - Client exits - error = ", strerror(-err));
				dev->status = STAT_ABORT;
			}
		}

		if (err >= 0)
			/* check paramter from command line */
			err = check_parms(dev);
		else
			dev->status = STAT_ABORT;

		if (err >= 0) {
			/* the storage exists and could be opened */
			if (dev->status == STAT_OK) {
				/* call ioctls for raw block devices or flash storages */
				if ((dev->type == BLOCK_FLASH_RAW) || 
					(dev->type == BLOCK_FLASH) ||
					(dev->type == BLOCK_RAW))
					err = get_device_dimensions(dev);
				
				if (err >= 0)
					/* get the header */
					err = read_dev_header(dev);
				else
					dev->status = STAT_ABORT;

				if (err >= 0)
					err = check_dev_header(dev);
				else
					dev->status = STAT_ABORT;

				if (err >= 0)
					err = check_system_view(dev);
				else
					dev->status = STAT_ABORT;

				if (err < 0)
					dev->status = STAT_ABORT;
			}
		}
			
		if(err >= 0) {
			if (dev->access_mode != O_RDONLY){
				if (dev->status == STAT_OK) {
					if (dev->command) {
						if (!strcmp(dev->command, "erase"))
							dev->status = STAT_EX_ERASE;
					}
				}
			}
 
			if ((dev->w_width < DEFAULT_ALIGNMENT) || (!~dev->w_width)) {
				if (dev->type != BLOCK) {
					err = -EINVAL;
					dev->status = STAT_ABORT;
					syslog(LOG_CRIT, "%s %s", " Write size not set for raw block or flash access. "
						   "Device will be ignored: ",dev->name);
				} else
					dev->w_width = DEFAULT_ALIGNMENT;
			}

			if ((dev->status == STAT_CREATE) && (!dev->header)) {
				uint32_t len = aligned_len(sizeof(ErrmemBackendBlockDeviceHeader_t), dev->w_width);
				dev->header = (ErrmemBackendBlockDeviceHeader_t*)get_mem(len);
				if (!dev->header) {
					err = -ENOMEM;
					dev->status = STAT_ABORT;
				} else
					memset(dev->header, 0xFF, len);
			}
				
			if ((err >= 0) && dev->blocksize) {
				dev->block = (ErrmemBackendBlockDeviceBlock_t*)
					get_mem(dev->blocksize);
				if (!dev->block) {
					err = -ENOMEM;
					dev->status = STAT_ABORT;
				} else
					memset(dev->block, 0xFF, dev->blocksize);
			}
			
			if (dev->type == BLOCK_NOR_RAW) {
				if ((err >= 0) && dev->wait && dev->erasesize) {
					/* Determine number of eraseblocks and create times array if necessary */
					uint32_t no_eblocks = (dev->blocks * (dev->blocksize / dev->erasesize));
					dev->times = (uint64_t*)get_mem(no_eblocks * sizeof(uint64_t));
					if (!dev->times) {
						err = -ENOMEM;
						dev->status = STAT_ABORT;
					}
				}
			}

			if (err >= 0)
				err = setup_dev(dev);
		}

		if (err >= 0) {
			/* transfer header data into block section */
			*(ErrmemBackendBlockDeviceHeader_t*)((void*)dev->block) = *dev->header;
			if (dev->status == STAT_OK) {
				/* re-use an existing persistent error memory */
				dev->seed         = dev->header->crc;
				dev->blockmagic   = dev->header->blockmagic;
				dev->w_width      = dev->header->w_width;
			} else if (dev->status == STAT_ERASED)
				/* This is because we may be erased here but we didn't call the
				 * erase function. The blockmagic in the header section is valid
				 * for the current storage at this point in any case
				 */
				dev->blockmagic = dev->header->blockmagic;
		}

		/*calculate base for message magic */
		if (err >= 0)
			dev->msg_magic = getCurrentMsgMagic(dev->blockmagic);

		/* we are looking at the very first block. Therefore l->block->magic has to be MAGIC_HEADER */
		if ( (err >= 0) && (dev->header->magic == MAGIC_HEADER)) {
			uint32_t b       = dev->blocks;
			uint32_t cur     = 0;
			uint32_t max     = 0;
			uint32_t i       = 0;
			uint32_t idx     = 0;

			for (; i < b; i++) {
				cur = get_blocknum(dev, i);
				if (~cur){
 					if(cur > max) {
						max = cur;
						idx = i;
					}
					dev->used++;
				}
			}
			/* Restore the file handle including O_SYNC */
			(void)close_dev(dev->fd, dev->name);
			dev->fd = open_dev(dev->name, dev->access_mode, 0);
			if (dev->fd < 0) {
				err = dev->fd;
				dev->fd = -1;
				dev->status = STAT_ABORT;
			}

			if (err >= 0) {
				dev->blocknum = max;
				err = seek_dev(dev->fd, dev->name, idx*dev->blocksize, SEEK_SET);
			}

			if (err >= 0) {
				uint32_t pos = err;
				err = read_dev(dev->fd, dev->name, (char*)dev->block, dev->blocksize);
				if (err >= 0) {
					dev->writepos =
						get_write_position((ErrmemBackend_t*)dev);
					/* position file pointer to the current write position
					 * always keep in mind that dev->writepos is index in data field
					 * and not the offset from block start!
					 */
					pos += offsetof(ErrmemBackendBlockDeviceBlock_t, data) + 
						dev->writepos;
					if (dev->block->magic == MAGIC_HEADER)
						pos += aligned_len(sizeof(ErrmemBackendBlockDeviceHeader_t),
										   dev->w_width);
					err = seek_dev(dev->fd, dev->name, pos, SEEK_SET);
					if (err < 0)
						syslog(LOG_CRIT, "%s %d %s %s",
							   "failed to seek to final position during "
							   "creation - pos = ", pos, " - error: ",
							   strerror(-err));
					else
						dev->writepos_absolute = err;
				}
			} else
				syslog(LOG_CRIT, "%s %d %s %s", "failed to seek to pos", idx*dev->blocksize, " - error: ", strerror(-err));
		}
	} else {
		syslog(LOG_CRIT, "%s" ,"create (block device backend): parameter check for dev failed");
		err = -EINVAL;
	}
	return err;
}

/* This function searches the block in the storage with the highest
 * blocknumber and returns the offset in bytes in the file. 
 * That is (index of block * blocksize).
 * Note: Be aware of the fact that the cursor position in the file assigned 
 * with the file handle which comes with the backend structure (si->backend->fd not si->fd)
 * may be changed during this function call because of the call to get_blocknum.
 * Function get_blocknum uses the file handle of the passed backend structure to read
 * the block of given index.
 * If necessary you have to take care of the cursor position on your own.
 */
static uint32_t get_current_pos(SessionInfo_t* si) 
{
	uint32_t rc      =  0;
	uint32_t cur     =  0;
	uint32_t max     =  0;
	uint32_t idx     =  0;

	if (si && si->backend) {
		ErrmemBackendBlockDevice_t* b = (ErrmemBackendBlockDevice_t*)si->backend;
		uint32_t i                    = b->persistent;

		/* nothing to do because we only have persistent blocks */
		if (b->blocks == b->persistent)
			rc = ~0;

		if (!rc) {
			for (; i < b->blocks; i++) {
				cur = get_blocknum(b, i);
				if (~cur && cur > max) {
					max = cur;
					idx = i;
				}				
			}
			rc = idx * b->blocksize;
		}
		if (max && si->seq_num && si->seq_num != max) {
			si->seq_num = max;
			b->blocknum  = max;
		}
	}
	return rc;
}

/* This function searches for the oldest message in the persistent storage.
 * The persistent part is handled outside this function because these
 * blocks are constant in a used storage. They will never be overwritten
 * during one life cycle of the storage. This function is called when the
 * persistent part is done.
 * It is to distinguish between daemon mode and client (application) mode.
 * If this code is executed by the daemon, the backend (si->backend or b) knows
 * the current values. Using its file handle or block information we always
 * have the current inforamtion.
 * This is completely different when this code is executed in client mode.
 * In this case we only have the current inforamtion from that point in time
 * where the backend information was initialized (create_backend)
 * If the storage is modified in between we will not get this information from
 * the backend structure. We have to check that manually.
 * Therefore this functionaltiy is implemented in separated functions.
 * The potentially oldest message is always supposed to be in the current block
 * which is in use at a higher addresses than the current write pointer points to.
 * This is the case when we already had a wrap around.
 * Otherwise if we will not found older messages in this location the oldest
 * message is the first one of the first block behind the persistent block area. 
 * 
 * -----------------------------------------------------------------
 * |    P0   |   P1   |   B1   |   B2   |   B3   |   B4   |   B5   |
 * -----------------------------------------------------------------
 * <- persisten area->                     ^
 *                                         | 
 *                                        cpos    Potential oldest message
 *                                                at the right of cpos in B3.
 *                                                If no valid message can be
 *                                                found the first message of
 *                                                B1 is the oldest one.
 * The position cpos has to be determined in every call to this function
 * because the storage may have been modified in between.
 *
 * Note: It is not necessary to lock the device during this operation because the 
 *       deamon is single threaded on the socket interface. Nobody can write into
 *       the persistent storage while a message is requested from it. But between
 *       two read requests the storage can be modified. This has to be taken into
 *       account.
 */
static uint32_t get_block_have_cur_oldest_msg(SessionInfo_t* si)
{
	uint32_t rc   = ~0;
	int32_t  cpos =  0; /* current cursor postion in the storage */
	uint32_t c    =  0; /* current block in use in the storage   */
	if (si && si->backend) {
		ErrmemBackendBlockDevice_t* b = (ErrmemBackendBlockDevice_t*)si->backend;
		/* do we already have messages outside the persistent area ? Otherwise we
		 * have nothing to do here */
		if (b->blocknum >= b->persistent) {
			/* get the current cursor position from the real backend */
			if (!si->client_mode)
				cpos = seek_dev(b->fd, b->name, 0, SEEK_CUR);
			else 
				cpos = get_current_pos(si);
			if (cpos >= 0) {
				/* the current block in use may still have messages which were written before a wrap around */
				c = (uint32_t)cpos / b->blocksize;
				/* If this condition is not true we are somewhere outside the existing persistent storage.
				 * This is a disaster */
				if (c < b->blocks) {
					/* are we really outside the persistent area? */
					if (c >= b->persistent) {
						/* just now finished reading the persistent storage area */
						if (!~si->num || (~si->num && b->persistent && si->num == b->persistent-1)) {
							/* no wrap around occurd in the persistent storage */
							if (c == b->blocknum) {
								si->num       = b->persistent; /* index of block to use         */
								si->num_start = c;             /* index of current block in use */
								si->seq_start = b->blocknum;   /* blocknumber of current block in use */
								si->block_cur = 0;             /* flag to instruct algorithm to check for old messages */
							} else {
								/* wrap around occured already - the current block has to be
								 * checked for old messages which are still available and not
								 * be overwritten
								 */
								si->num       = c;
								si->num_start = c;
								si->seq_start = b->blocknum;
								si->block_cur = 1;
							}
						} else {
							/* if the current block has the same index in comparison to the last call
							 * we additionally have to check the block number because the storage
							 * may be completely overwritten between these two calls to this function
							 */
							if (si->num_start == c) {
								if (si->seq_start == b->blocknum) {
									si->num++;
									if (si->num == b->blocks)
										si->num = b->persistent;
									si->block_cur = 0;
								} else {
									si->num       = c;
									si->num_start = c;
									si->seq_start = b->blocknum;
									si->block_cur = 1;
								}									
							} else {
								/* now the number of overwritten blocks has to be checked 
								 * the write pointer has changed to some block but we don't
								 * know whether it overtakes our current read pointer. This
								 * we can check by calculating the number of overwritten
								 * blocks and the number of blocks our current read pointer is
								 * ahead of the write pointer */
								uint32_t pn = si->num + 1;                       /* possible next block */
								uint32_t ov = b->blocknum - si->seq_start;  /* number of overwritten blocks */
								if (pn == b->blocks)
									pn = b->persistent;
								if (pn == c) {
									si->num       = c;
									si->num_start = c;
									si->seq_start = b->blocknum;
									si->block_cur = 1;
								} else if (pn > c) {
									if (ov < (pn -c)) {
										si->num++;
										if (si->num == b->blocks)
											si->num = b->persistent;
										si->num_start = c;
										si->seq_start = b->blocknum;
										si->block_cur = 0;
									} else {
										si->num       = c;
										si->num_start = c;
										si->seq_start = b->blocknum;
										si->block_cur = 1;
									} 									
								} else if (pn < c) {
									if (ov < ((b->blocks - b->persistent) + (pn - b->persistent))) {
										si->num++;
										if (si->num == b->blocks)
											si->num = b->persistent;
										si->num_start = c;
										si->seq_start = b->blocknum;
										si->block_cur = 0;
									} else {
										si->num       = c;
										si->num_start = c;
										si->seq_start = b->blocknum;
										si->block_cur = 1;
									}
								}
									
							}
						}
						rc = si->num;
					}
				}
			}
		}
	}
	return rc;
}

static uint32_t get_msg_from_block(SessionInfo_t* si, 
								   ErrmemBackendBlockDeviceBlock_t* block, 
								   uint32_t hsize, int32_t begin_block)
{
	uint32_t rc = 0;
	if (si && block) {
		ErrmemBackendBlockDevice_t* b = (ErrmemBackendBlockDevice_t*)si->backend;
		PersistentMessage_t pmsg      = {0};
		PersistentMessage_old_t opmsg = {0};
		struct errmem_message* e_msg  = NULL;
		uint32_t endpos               = 0;
		uint32_t clen                 = 0;
		uint32_t crc                  = 0;
		int32_t  found                = 0;
		uint16_t p_crc                = 0;
		uint32_t magic                = 0;
		uint32_t offset               = 0;
		uint32_t blocknum             = 0;
		uint32_t len                  = 0;
		uint32_t acc                  = 0;

		magic = *(mkuint32(&block->data[si->pos], 0));
		/* Check whether CRC for message header is supported or not */
		if (magic == MAGIC_MESSAGE) {
			/* This message format does not support CRC check but the
			 * message is a valid one 
			 */
			opmsg = *(mkpmsgo(&block->data[si->pos], 0));
			offset   = offsetof(PersistentMessage_old_t, msg);
			e_msg    = mkmsg(block->data, si->pos + offset);
			blocknum = opmsg.blocknum;
			len      = opmsg.len;
			acc = 1;
		} else if (magic == b->msg_magic) {
			pmsg = *(mkpmsg(&block->data[si->pos], 0));
			p_crc = pmsg.crc;
			pmsg.crc = 0;
			pmsg.crc = calc_crc16(b->seed, (void*)&pmsg, offsetof(PersistentMessage_t, msg));
			if (pmsg.crc == p_crc) {
				/* valid message with new format */
				offset   = offsetof(PersistentMessage_t, msg);
				e_msg    = mkmsg(block->data, si->pos + offset);
				blocknum = pmsg.blocknum;
				len      = pmsg.len;
				acc = 1;
			}
		}

		if (acc) {
			if (si->block_cur) {
				/* Search the block for old messages written into
				 * this block the last time this block was used. 
				 * But only the last time, not the second or third 
				 * but last. This can happen if the space in a block
				 * at the end is not enough to store a new message which
				 * normally would overwrite an old stored message. This new
				 * message will be written into the next block while the
				 * old message remains in this block. 
				 */
				if (blocknum == si->seq_num) {
					si->seq_next = blocknum;
					if (~si->seq_new && (si->seq_new != blocknum))
						si->seq_new  = blocknum;
					found = 1;
				}
			} else {
				/* we have a message from the current block here */
				if (blocknum == block->blocknum)
					found = 1;
			}
		}

		if (found && e_msg) {
			if (len) {
				/* check length against message length */
				if (len > sizeof(struct errmem_message))
					clen = sizeof(struct errmem_message);
				else
					clen = len;
				/* check length against the blocksize 
				 * first byte of struct PersistentMessage is at position s->pos = 0 */
				endpos = hsize + si->pos + offset + clen;
				if (endpos > b->blocksize)
					clen = clen - (endpos - b->blocksize);

				/* check the crc */
				crc = e_msg->internal.crc;
				e_msg->internal.crc = 0;
				e_msg->internal.crc = calc_crc16(b->seed, (void*)e_msg, clen);
				if (e_msg->internal.crc != crc)
					e_msg->internal.flags |= ERRMEM_FLAG_CRC_ERROR;
				/* copy the message */
				memcpy(&si->msg_queue->msg, e_msg, clen);
				si->msg_queue->next = NULL;
				/* if the copied length is not the given message length set the imcomplete flag */
				if (len != clen)
					si->msg_queue->msg.internal.flags |= ERRMEM_FLAG_INCOMPLETE;
				/* the second message in a block has to be determined by calculating the 
				 * alignment over the leading block meta data information */
				if (!begin_block)
					si->pos += aligned_len((offset + clen), b->w_width);
				else
					si->pos  = aligned_len((hsize + offset + clen), b->w_width) - hsize;
			} else
				rc = ~0;
		} else {
			/* set position to a possible next index which has to be an aligned one
			 * hsize at the beginning of a block contains already at last byte si->pos = 0 
			 */
			if(!si->pos && b->w_width)
				si->pos = aligned_len(hsize, b->w_width) - hsize;
			else
				si->pos += b->w_width;
			rc = ~0;
		}
	}
	return rc;
}

static void read_message(SessionInfo_t* si)
{
	int32_t err         = 0;
	int32_t begin_block = 0;
	int32_t next_req    = 0;
	if (si && si->block && si->backend) {
		ErrmemBackendBlockDevice_t* b = (ErrmemBackendBlockDevice_t*)si->backend;
		ErrmemBackendBlockDeviceBlock_t* block = (ErrmemBackendBlockDeviceBlock_t*)si->block;
		uint32_t hsize = sizeof(ErrmemBackendBlockDeviceBlock_t) - 1; /* Last byte already payload */
		uint32_t len   = 0;
		uint32_t cpos  = 0;
		if (block->magic == MAGIC_HEADER) {
			len   = aligned_len(b->header->header_length, b->w_width);
			block = (ErrmemBackendBlockDeviceBlock_t*)addptr((unsigned char*)block, len);
			hsize += len;
		}

		if (!si->pos){
			begin_block = 1;
			// check for the currently valid block magic
			if (block->magic == si->block_magic) {
				// check the crc
				uint16_t crc = block->crc;
				block->crc = 0;
				if (calc_crc16(b->seed, (void*) block,
							   sizeof(ErrmemBackendBlockDeviceBlock_t))==crc) {

					if (si->block_cur) {
						/* Set blocknum for current block to the blocknum this
						 * block had before it was overwritten - we assume
						 * old messages from that time. Older messages. e.g.
						 * from another wrap before this wrap around will be
						 * ignored. These messages will stand alone without any
						 * context.
						 * In case we don't find messages we continue the read session with the
						 * next higher blocknum. 
						 */
						if (block->blocknum >= b->blocks) {
							int32_t bwrap = (int32_t)(block->blocknum - (b->blocks - b->persistent));
							si->seq_num = bwrap;
						} else {
							/*no wrap around no old messages */
							si->block_cur = 0; 
							si->seq_num = block->blocknum;
						}
					} else
						si->seq_num = block->blocknum;

					/* if seq_new is -1 because of reset in create messages 
					 * store current next as last because we start to read a
					 * new block. Note that seq_last is only updated here if
					 * it has a valid value (>=0), In case it is -1 it will
					 * be updated in function create_messages. This is
					 * necessary to recognize whether a block 0 already
					 * has been overwritten in case of no persistent blocks.
					 * For sure this could also be done by additional flags but
					 * this solution is not implemented here */
				    if (~si->seq_last && !~si->seq_new)
						si->seq_last = si->seq_next;

					/* The current block with number si->blocknum is the next to read */
					si->seq_next = si->seq_num;
					si->seq_new  = si->seq_num;
					
					/* During read operation the blocks are overwritten */
					if (si->seq_next > si->seq_max) {
						si->seq_max = si->seq_next;
						si->status |= ERR_READ_CONTINUE;
					}
					/* This block has been read already - stop here */
					if (~si->seq_last && si->seq_next < si->seq_last) {
						si->state = ERRMEM_SESSION_READ_DONE;
						si->status = ERR_READ_END;
					}
				} else
					si->status = ERR_BLOCK_CORRUPTED;
			} else if (si->block_magic != b->blockmagic){
				si->state  = ERRMEM_SESSION_BACKEND_ERASED;
				si->status = ERR_SESSION_BACKEND_ERASED;
			} else
				si->status = ERR_BLOCK_ERASED;
		}

		if (!(si->status & ~ERR_READ_CONTINUE)) {
			next_req = 1;
			do {
				cpos = si->pos;
				err = get_msg_from_block(si, block, hsize, begin_block);
				begin_block = 0;
				/* no valid message found */
				if (!~err) {
					/* here we can distinguish between invalid magic or invalid length 
					 * if si->pos > cpos the header magic is corrupted or invalid 
					 * we are searching the complete block for other messages but we
					 * cannot determine whether we will found old messages */
					if (si->pos == cpos)
						si->pos += b->w_width;
					
					if (si->pos + hsize + offsetof(struct PersistentMessage, msg) + 
						offsetof(struct errmem_message, message) >= b->blocksize) {
						si->status |= ERR_BLOCK_END;
						si->pos = 0;
						next_req = 0;
					}
				}
				else
					/* valid message found */
					next_req = 0;
			} while (next_req);
		}
	} else if (si)
		si->status = -EIO;
	begin_block = 0;
}

static void get_new_block(SessionInfo_t* si)
{
	int32_t err = 0;
	if (si && si->block && si->backend) {
		if (si->status & ERR_BLOCK_END){
			ErrmemBackendBlockDevice_t* b = (ErrmemBackendBlockDevice_t*)si->backend;
			err = seek_dev(si->fd, b->name, si->num*b->blocksize, SEEK_SET);
			if (err >= 0)
				err = read_dev(si->fd, b->name, si->block, b->blocksize);
			if (err >= 0)
				si->status = ERR_OK;
		}
	}  else if (si) {
		si->status = -ENOTRECOVERABLE;
	}
}

static void get_msg(SessionInfo_t* si)
{
	if (si && si->backend){
		ErrmemBackendBlockDevice_t* b = (ErrmemBackendBlockDevice_t*)si->backend;
		/* get memory for a block in case we are starting to read */
		if (!~si->num){
			si->block = get_mem(b->blocksize);
			if (!si->block)
				si->status = -ENOMEM;
			else
				si->status |= ERR_BLOCK_END;
		}
		if ((si->status >= 0) && 
			((si->status == ERR_OK) || 
			 (si->status & ERR_BLOCK_END) || 
			 (si->status & ERR_READ_CONTINUE))) {
			/* we have to repeat this because not until read_message is executed
			 * we don't know whether the block is done 
			 */
			do {
				/* In case we are a daemon we alway get the real time information 
				 * In case we are an application this is constant 
				 */
				si->seq_max = b->blocknum;
				if (si->state == ERRMEM_SESSION_READ_PERSISTENT) {
					si->num = 0;
					si->pos = 0;
					si->block_cur = 0; /* persistent blocks will never be overwritten in a life cycle */
					si->state = ERRMEM_SESSION_READING_PERSISTENT;
				}
				if ((si->state == ERRMEM_SESSION_READ) && 
					(si->status >= 0) && 
					(si->status == ERR_BLOCK_END)) {
					si->num = get_block_have_cur_oldest_msg(si);
					si->pos = 0;
					if (!~si->num)
						si->status = -EFAULT;
				}
				
				if ((si->status >= 0) && (si->status & ERR_BLOCK_END))
					get_new_block(si);
				if ((si->status >= 0) && (si->status & ERR_BLOCK_END))
					/* This block couldn't be read, try next and assume 
					 * block number increases by one */
					si->seq_num++;
				if ((si->status >= 0) && 
					((si->status == ERR_OK) || (si->status & ERR_READ_CONTINUE)))
					read_message(si);
				
				/* implement criteria for read fail in get_new _block here because
				 * read_message will not be called then */
				if ((si->status >= 0) && (si->status & ERR_BLOCK_END)) {
					/* reading is done - no more messages */
					if (!(si->status & ERR_READ_CONTINUE) && (si->seq_num >= si->seq_max))
						si->state = ERRMEM_SESSION_READ_DONE;
					else {
						/* switch from persistent blocks to non persistent blocks */
						if (si->state == ERRMEM_SESSION_READING_PERSISTENT &&
							si->num + 1 >= b->persistent) {
							si->state = ERRMEM_SESSION_READ;
						} else if (si->state == ERRMEM_SESSION_READING_PERSISTENT)
							si->num++;
					}
					si->status &= ~ERR_READ_CONTINUE;
				} else if ((si->status >= 0) &&
						   ((si->status == ERR_BLOCK_ERASED) || (si->status == ERR_BLOCK_CORRUPTED))) {
					if (si->num + 1 >= b->persistent)
						si->state = ERRMEM_SESSION_READ;
					si->status = ERR_BLOCK_END;
					if (si->num == si->num_start)
						si->state = ERRMEM_SESSION_READ_DONE;
				}
				if (si->state == ERRMEM_SESSION_READ_DONE)
					si->status = ERR_READ_END;
			} while (si->status == ERR_BLOCK_END);
		}
	} else if (si) {
		si->state  = ERRMEM_SESSION_BACKEND_ERASED;
		si->status = ERR_SESSION_BACKEND_ERASED;
	}
}


static void open_session(SessionInfo_t* si)
{
	if (si && si->backend){
		if (((ErrmemBackendBlockDevice_t*)si->backend)->name) {
			si->fd = open(((ErrmemBackendBlockDevice_t*)si->backend)->name,
						  O_RDONLY);
			if (!~si->fd)
				si->status = -errno;
		} else
			si->status = -EINVAL;
	} else if (si)
		si->status = -ENOTRECOVERABLE;
}

static void read_session(SessionInfo_t* si)
{
	if (si && si->backend) {
		ErrmemBackendBlockDevice_t* b = (ErrmemBackendBlockDevice_t*)si->backend;
		if (b->status == STAT_OK) {
			if (si->state == ERRMEM_SESSION_ESTABLISHED){
				si->block_magic = b->blockmagic;
				si->persistent  = b->persistent;
				if (si->persistent > 0)
					si->state = ERRMEM_SESSION_READ_PERSISTENT;
				else
					si->state = ERRMEM_SESSION_READ;
			}
			if (si->block_magic == b->blockmagic)
				get_msg(si);
			else {
				si->state  = ERRMEM_SESSION_BACKEND_ERASED;
				si->status = ERR_SESSION_BACKEND_ERASED;
			}
		} else {
			si->state  = ERRMEM_SESSION_BACKEND_ERASED;
			si->status = ERR_SESSION_BACKEND_ERASED;
		}
	}
}

ErrmemBackend_t* create_block_structure(ErrmemBackendDev_t* p_dev)
{
	int32_t err = 0;
    ErrmemBackendBlockDevice_t* dev = NULL;

	if (p_dev) {
		dev = (ErrmemBackendBlockDevice_t*)
			get_mem(sizeof(ErrmemBackendBlockDevice_t));

		if (dev) {
			// setup the error memory backend interface
			dev->backend.get_command   = get_command;
			dev->backend.get_name      = get_name;
			dev->backend.get_nr        = get_nr;
			dev->backend.set_nr        = set_nr;
			dev->backend.is_default    = is_default;
			dev->backend.set_default   = set_default;
			dev->backend.get_type      = get_type;
			dev->backend.info          = info;
			dev->backend.get_last_seq_nr = get_last_seq_nr;
			dev->backend.dump          = dump;
			dev->backend.erase         = erase;
			dev->backend.store         = store;
			dev->backend.create        = create;
			dev->backend.destroy       = destroy;
			dev->backend.open_session  = open_session;
			dev->backend.read_session  = read_session;
			dev->nr                    = p_dev->nr;
			dev->type                  = p_dev->type;
			dev->access_options        = p_dev->access_options;
			dev->access_mode           = 0;
			dev->status                = STAT_OK;
			dev->blocknum              = 0;
			dev->seqnum                = ~0;
			dev->fd                    = -1;
			dev->erasesize             = ~0;
			dev->w_width               = p_dev->w_width;
			dev->blocksize             = p_dev->size;
			dev->blocks                = p_dev->cnt;
			dev->persistent            = p_dev->persistent;
			dev->last_stored           = 0;
			dev->stat_mode             = 0;
			dev->stat_blocksize        = 0;
			dev->stat_blocks           = 0;
			dev->stat_size             = 0;
			dev->msg_magic             = 0;
			dev->wait                  = p_dev->wait;
			dev->retry                 = p_dev->retry;

			dev->block                 = NULL;
			dev->header                = NULL;
			dev->times                 = NULL;

			if (dev->nr == 1)
				dev->def = 1;
			else
				dev->def = 0;

			if (p_dev->name) {
				dev->name = strndup(p_dev->name, strlen(p_dev->name) + 1);
				dev->name = dev->name;
				if (!dev->name)
					err = -ENOMEM;
			}

			if (p_dev->command) {
				dev->command = strndup(p_dev->command, strlen(p_dev->command) + 1);
				if (!dev->command)
					err = -ENOMEM;
			}

			/* if alignement is known here get the header */
			if (!err && dev->w_width && ~dev->w_width) {
				uint32_t len = 0;
				if (dev->w_width < DEFAULT_ALIGNMENT)
					dev->w_width = DEFAULT_ALIGNMENT;
				len = aligned_len(sizeof(ErrmemBackendBlockDeviceHeader_t), dev->w_width);
				dev->header = (ErrmemBackendBlockDeviceHeader_t*)get_mem(len);
				if (!dev->header)
					err = -ENOMEM;
				else
					memset(dev->header, 0xFF, len);
			}
			if (err < 0) {
				if (dev->name)
					syslog(LOG_CRIT, "%s %s %s %s", 
						   "Backend: ", dev->name,
						   "could not be created - error: ", strerror(-err));
				else
					syslog(LOG_CRIT, "%s %s", 
						   "A block device backend could not be created - "
						   "error: ", strerror(-err));
				destroy(&dev->backend);
			} else
				return &dev->backend;
		} else
			err = -ENOMEM;
	} else
		err = -EINVAL;
	
	return NULL;
}
